<?php

namespace Dealerdirect\VehicleClassification\Tests;

use Dealerdirect\VehicleClassification\Models\FuelUtils;
use Dealerdirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use Dealerdirect\VehicleClassification\Models\Vehicle;
use Dealerdirect\VehicleClassification\VehicleClassification;
use PHPUnit\Framework\TestCase;
use Dealerdirect\VehicleClassification\Rules\Motors\RuleSet as MotorRuleSet;
use Dealerdirect\VehicleClassification\Rules\Cars\RuleSet as CarRuleSet;

/**
 * Class VehicleClassificationClassTest
 * @package Dealerdirect\VehicleClassification\Tests
 */
class VehicleClassificationTest extends TestCase
{
    /**
     * @param $vehicleData
     *
     * @dataProvider provideDataGroupMotorYoung
     */
    public function testVehicleClassificationAllOnMotorGroupYoung(array $vehicleData): void
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = new VehicleClassification(new MotorRuleSet());
        $classifiedId = $vehicleClassification->classifyVehicle($vehicle);

        $this->assertEquals(MotorRuleSet::GROUP_YOUNG_ID, $classifiedId);
    }

    /**
     * @param $vehicleData
     *
     * @dataProvider provideDataGroupMotorOther
     */
    public function testVehicleClassificationAllOnMotorGroupOther(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = new VehicleClassification(new MotorRuleSet());
        $classifiedId = $vehicleClassification->classifyVehicle($vehicle);

        $this->assertEquals(MotorRuleSet::GROUP_OTHER_ID, $classifiedId);
    }

    /**
     * @param $vehicleData
     *
     * @dataProvider provideDataGroupCarExport
     */
    public function testVehicleClassificationAllOnCarGroupExport(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = new VehicleClassification(new CarRuleSet());
        $classifiedId = $vehicleClassification->classifyVehicle($vehicle);

        $this->assertEquals(CarRuleSet::GROUP_EXPORT_ID, $classifiedId);
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupMotorYoung()
    {
        $currentYear = (int)date('Y');
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 14507,
                'makeName' => 'PIAGGIO',
                'modelName' => 'M64',
                'yearOfManufacture' => $currentYear - 8,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4750
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 22000,
                'makeName' => 'YAMAHA',
                'modelName' => 'XT 660 R',
                'yearOfManufacture' => $currentYear - 6,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 3300
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 19700,
                'makeName' => 'DUCATI',
                'modelName' => 'MONSTER 796',
                'yearOfManufacture' => $currentYear - 6,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 6000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 11000,
                'makeName' => 'BENZHOU',
                'modelName' => 'IVA LUX50',
                'yearOfManufacture' => $currentYear - 6,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 650
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 25850,
                'makeName' => 'HONDA',
                'modelName' => 'NC700D',
                'yearOfManufacture' => $currentYear - 8,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4250
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 38000,
                'makeName' => 'BMW',
                'modelName' => 'R 1150 GS',
                'yearOfManufacture' => $currentYear - 8,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 0
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 82136,
                'makeName' => 'HARLEY-DAVIDSON',
                'modelName' => 'FLHT - FLHTI ELECTRA GLIDE STANDARD ',
                'yearOfManufacture' => $currentYear - 21,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4400
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 30260,
                'makeName' => 'BMW',
                'modelName' => 'R 850 R',
                'yearOfManufacture' => $currentYear - 5,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4000
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupMotorOther()
    {
        $currentYear = (int)date('Y');
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '2',
                'mileage' => 3600,
                'makeName' => 'PIAGGIO',
                'modelName' => 'FLY',
                'yearOfManufacture' => $currentYear - 11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 75
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 14000,
                'makeName' => 'YAMAHA',
                'modelName' => 'VM04',
                'yearOfManufacture' => $currentYear - 9,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 6500
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 15000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'M34',
                'yearOfManufacture' => $currentYear - 13,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 2800
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 79000,
                'makeName' => 'BMW',
                'modelName' => '328 IA',
                'yearOfManufacture' => $currentYear - 12,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 69999,
                'makeName' => 'HONDA',
                'modelName' => 'PAN EUROPEAN',
                'yearOfManufacture' => $currentYear - 12,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 5000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 75000,
                'makeName' => 'HONDA',
                'modelName' => 'VALKYRIE INTERSTATE',
                'yearOfManufacture' => $currentYear - 10,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 10000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 45000,
                'makeName' => 'YAMAHA',
                'modelName' => 'VM04',
                'yearOfManufacture' => $currentYear - 11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 6500
            )),
            // Classics:
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 89000,
                'makeName' => 'YAMAHA',
                'modelName' => 'Classic1',
                'yearOfManufacture' => $currentYear - 36,
                'classic' => '1',
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 2500
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 89000,
                'makeName' => 'HONDA',
                'modelName' => 'Classic2',
                'yearOfManufacture' => 1976,
                'classic' => '0',
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 1500
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupMotorDemolition()
    {
        $currentYear = (int)date('Y');
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 62000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'FLY',
                'yearOfManufacture' => $currentYear - 11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 75
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 82000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'FLY',
                'yearOfManufacture' => $currentYear - 11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 75
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 100000,
                'makeName' => 'YAMAHA',
                'modelName' => 'VM04',
                'yearOfManufacture' => $currentYear - 12,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4500
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 81000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'M34',
                'yearOfManufacture' => $currentYear - 19,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 2800
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 62000,
                'makeName' => 'YAMAHA',
                'modelName' => '100',
                'yearOfManufacture' => 1984,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 750
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupCarExport()
    {
        return array(
            array(array(
                'id' => '1',
                'licensePlate' => '44NJTB',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 316000,
                'makeName' => 'SAAB',
                'modelName' => '9-3',
                'yearOfManufacture' => ((int)date('Y') - 15),
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => '10156',
                'numberOfDoors' => 5,
                'askingPrice' => '1650'
            ))
        );
    }
}
