<?php

namespace Dealerdirect\VehicleClassification\Models;

/**
 * Class Fuel
 *
 * Static class, used for defining vehicle fuel type
 *
 * @package Dealerdirect\VehicleClassification\Models
 */
class FuelUtils
{
    const UNKNOWN = 0;
    const GASOLINE = 1;
    const DIESEL = 2;
    const ELECTRICITY = 3;
    const LPG = 4;
    const LPG_G3 = 5;
    const CRYOGENIC = 6;
    const CNG = 7;
    const HYDROGEN = 8;
    const HYBRID_GASOLINE = 9;
    const HYBRID_DIESEL = 10;

    /**
     * Get integer fuel type value for given fuel type as string
     *
     * @param string $fuelType The fuel type as string (English name)
     *
     * @return int The fuel type as integer value
     *
     */
    public static function getFuelTypeAsInt($fuelType)
    {
        switch (strtolower($fuelType)) {
            case 'gasoline':
                return self::GASOLINE;
            case 'diesel':
                return self::DIESEL;
            case 'electricity':
                return self::ELECTRICITY;
            case 'lpg':
                return self::LPG;
            case 'g3':
            case 'lpg g3':
            case 'lpg-g3':
                return self::LPG_G3;
            case 'cryogenic':
                return self::CRYOGENIC;
            case 'cng':
                return self::CNG;
            case 'hydrogen':
                return self::HYDROGEN;
            case 'hybrid-gasoline':
                return self::HYBRID_GASOLINE;
            case 'hybrid-diesel':
                return self::HYBRID_DIESEL;
        }

        return self::UNKNOWN;
    }

    /**
     * Get string representation for given fuel type integer.
     *
     * @param int $fuelType The fuel type
     *
     * @return string The fuel type string name (in English)
     */
    public static function getFuelTypeAsString($fuelType)
    {
        switch ($fuelType) {
            case self::GASOLINE:
                return 'gasoline';
            case self::DIESEL:
                return 'diesel';
            case self::ELECTRICITY:
                return 'electricity';
            case self::LPG:
                return 'lpg';
            case self::LPG_G3:
                return 'lpg-g3';
            case self::CRYOGENIC:
                return 'cryogenic';
            case self::CNG:
                return 'cng';
            case self::HYDROGEN:
                return 'hydrogen';
            case self::HYBRID_GASOLINE:
                return 'hybrid-gasoline';
            case self::HYBRID_DIESEL:
                return 'hybrid-diesel';
        }
        return '';
    }
}
