<?php

namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\VehicleTypeMotorBodyType as BodyType;
use DealerDirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\VehicleRuleSet;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use Ruler\Context;

class MotorsRuleSetTest extends \PHPUnit_Framework_TestCase
{

    /**
     * @var VehicleRuleSet
     */
    private $ruleSet;

    protected function setUp()
    {
        $this->ruleSet = new MotorsRuleSet();
    }


    /**
     * Test motor ruleset
     *
     * @param array $vehicleData The vehicle as array
     * @param int $expectedGroupId
     *
     * @dataProvider provideExpectedData
     */
    public function testMotorRuleSetShouldClassifySubjectsInExpectedGroup(array $vehicleData, $expectedGroupId)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $context = new Context($vehicleData);


        $this->assertEquals($expectedGroupId, $this->ruleSet->evaluate($context));
    }


    /**
     * @return array<array<array>>
     */
    public static function provideExpectedData()
    {
        $currentYear = (int)date('Y');
        return [
            // Scooters:
            [
                [
                    'engineCapacity' => 45,
                    'damageFree' => '2',
                    'mileage' => 3600,
                    'makeName' => 'PIAGGIO',
                    'modelName' => 'FLY',
                    'yearOfManufacture' => 2006,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 75
                ],
                GroupRulesInterface::GROUP_MOTORS_SCOOTER_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_SCOOTER,
                    'damageFree' => '2',
                    'mileage' => 3600,
                    'makeName' => 'PIAGGIO',
                    'modelName' => 'FLY',
                    'yearOfManufacture' => 2006,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 75
                ],
                GroupRulesInterface::GROUP_MOTORS_SCOOTER_ID
            ],
            // Trikes, Quads:
            [
                [
                    'engineCapacity' => 55,
                    'bodyType' => BodyType::QUAD,
                    'damageFree' => '2',
                    'mileage' => 3600,
                    'makeName' => 'PIAGGIO',
                    'modelName' => 'FLY',
                    'yearOfManufacture' => 2006,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 75
                ],
                GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'bodyType' => BodyType::TRIKE,
                    'damageFree' => '2',
                    'mileage' => 3600,
                    'makeName' => 'PIAGGIO',
                    'modelName' => 'FLY',
                    'yearOfManufacture' => 2006,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 75
                ],
                GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'bodyType' => BodyType::QUAD,
                    'damageFree' => '1',
                    'mileage' => 14000,
                    'makeName' => 'YAMAHA',
                    'modelName' => 'VM04',
                    'yearOfManufacture' => 2008,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 6500
                ],
                GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
            ],
            // Sidecar
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'bodyType' => BodyType::SIDE_CAR,
                    'damageFree' => true,
                    'mileage' => 90000,
                    'makeName' => 'BMW',
                    'modelName' => '328 IA',
                    'yearOfManufacture' => $currentYear-7,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 2000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // Honda Goldwing
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'bodyType' => BodyType::TOURING,
                    'damageFree' => '1',
                    'mileage' => 14000,
                    'makeName' => 'HONDA',
                    'modelName' => 'GOLDWING',
                    'yearOfManufacture' => 2008,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 6500
                ],
                GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
            ],
            // Pan European
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 75000,
                    'makeName' => 'HONDA',
                    'modelName' => 'PAN EUROPEAN',
                    'yearOfManufacture' => $currentYear-12,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 5000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // Goldwing
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 55000,
                    'makeName' => 'HONDA',
                    'modelName' => 'GOLDWING',
                    'yearOfManufacture' => $currentYear-16,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 4000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // Valkyrie
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 75000,
                    'makeName' => 'HONDA',
                    'modelName' => 'VALKYRIE INTERSTATE',
                    'yearOfManufacture' => $currentYear-10,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 10000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // FJR
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 75000,
                    'makeName' => 'YAMAHA',
                    'modelName' => 'FJR 1300',
                    'yearOfManufacture' => $currentYear-10,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 9000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // Wildstar
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 92000,
                    'makeName' => 'YAMAHA',
                    'modelName' => 'XV1600 WILDSTAR',
                    'yearOfManufacture' => $currentYear-13,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 9000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // Wildstar
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 76000,
                    'makeName' => 'YAMAHA',
                    'modelName' => 'XV1600 WILDSTAR',
                    'yearOfManufacture' => $currentYear-13,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 9000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // Midnight Star
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 56000,
                    'makeName' => 'YAMAHA',
                    'modelName' => 'XVS 1300 A MIDNIGHT STAR',
                    'yearOfManufacture' => $currentYear-10,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 11000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // BMW
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 99000,
                    'makeName' => 'BMW',
                    'modelName' => 'K1200 LT',
                    'yearOfManufacture' => $currentYear-14,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 3000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // Younger than 9 years old
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => '1',
                    'mileage' => 14507,
                    'makeName' => 'PIAGGIO',
                    'modelName' => 'M64',
                    'yearOfManufacture' => $currentYear-8,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 4750
                ],
                GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
            ],
            // 10. Is het voertuig ouder dan 25 jaar -> Overige
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 80000,
                    'makeName' => 'KAWASAKI',
                    'modelName' => 'VERSYS',
                    'yearOfManufacture' => $currentYear-26,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 1000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // 11. Is het voertuig jonger dan 15 jaar en heeft het minder dan 70.000km -> Overige
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 69000,
                    'makeName' => 'SUZUKI',
                    'modelName' => 'VERSYS',
                    'yearOfManufacture' => $currentYear-10,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 4000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // 12. Is het voertuig jonger dan 15 jaar -> Budget
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 71000,
                    'makeName' => 'TRIUMPH',
                    'modelName' => 'BONNEVILLE',
                    'yearOfManufacture' => $currentYear-14,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 3500
                ],
                GroupRulesInterface::GROUP_MOTORS_BUDGET_ID
            ],
            // 13. Is het voertuig jonger dan 20 jaar en heeft het minder dan 40.000km -> Overige
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 39000,
                    'makeName' => 'INDIAN',
                    'modelName' => 'CHIEFTAIN',
                    'yearOfManufacture' => $currentYear-19,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 8000
                ],
                GroupRulesInterface::GROUP_MOTORS_OTHER_ID
            ],
            // 14. Is het voertuig jonger dan 20 jaar en heeft het minder dan 70.000km -> Budget
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 69000,
                    'makeName' => 'DUCATI',
                    'modelName' => 'MONSTER 900',
                    'yearOfManufacture' => $currentYear-19,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 6000
                ],
                GroupRulesInterface::GROUP_MOTORS_BUDGET_ID
            ],
            // 15. Is het voertuig jonger dan 25 jaar en heeft het minder dan 60.000km -> Budget
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 59000,
                    'makeName' => 'KTM',
                    'modelName' => '125',
                    'yearOfManufacture' => $currentYear-24,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 4000
                ],
                GroupRulesInterface::GROUP_MOTORS_BUDGET_ID
            ],
            // 16. <Geen voorwaarde, alle overgebleven voertuigen> -> Schade & Sloop
            [
                [
                    'vehicleType' => Vehicle::TYPE_MOTOR,
                    'damageFree' => true,
                    'mileage' => 99000,
                    'makeName' => 'KTM',
                    'modelName' => '300',
                    'yearOfManufacture' => $currentYear-24,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => 500
                ],
                GroupRulesInterface::GROUP_MOTORS_DEMOLITION_ID
            ]
        ];
    }
}
