<?php

namespace Dealerdirect\VehicleClassification;

use Dealerdirect\Generic\Category\CallGroups;
use Dealerdirect\Generic\Category\CallGroupsMoto;
use DealerDirect\Generic\Category\Site;
use Exception;

enum VehicleClassificationIds: int
{
    /**
     * Vehicle Classification Id's
     *
     * Values are mapped to the id's in the database table:
     * dealer01_ddmain.vehicle_classification_descriptions
     */
    case CLASSIFICATION_NO_ID = 0;
    case CLASSIFICATION_YOUNG_ID = 1;
    case CLASSIFICATION_DEMOLITION_ID = 11;
    case CLASSIFICATION_RECREATIONAL_ID = 21;
    case CLASSIFICATION_TRANSPORT_ID = 31;
    case CLASSIFICATION_BUDGET_ID = 41;
    case CLASSIFICATION_EXPORT_ID = 51;
    case CLASSIFICATION_OTHER_ID = 61;
    case CLASSIFICATION_PREMIUM_ID = 71;

    public function toCallgroupId(int $reference, bool $returnAll = false): array|int
    {
        Site::assertExists($reference);

        //phpcs:ignore PHPCompatibility.Variables.ForbiddenThisUseContexts.OutsideObjectContext
        $mapping = $this->mapping();
        if (!array_key_exists($reference, $mapping)) {
            throw new Exception('Error, could not find reference');
        }

        $callgroupIds = $mapping[$reference];

        return $returnAll
            ? $callgroupIds
            : (int) $callgroupIds[array_rand($callgroupIds)];
    }

    private function mapping()
    {
        //phpcs:ignore PHPCompatibility.Variables.ForbiddenThisUseContexts.OutsideObjectContext
        return match ($this) {
            self::CLASSIFICATION_NO_ID => [
                Site::REF_CAR_NL => [CallGroups::NL_CAR_CHECKLIST],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_CHECKLIST],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_CHECKLIST],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_CHECKLIST],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_MARKTPLAATS],
                Site::REF_SCOOTER_NL => [CallGroupsMoto::NL_MOTO_CHECKLIST],
                Site::REF_MOTO_NL => [CallGroupsMoto::NL_MOTO_CHECKLIST],
                Site::REF_MOTO_BE => [CallGroupsMoto::BE_MOTO_CHECKLIST],
                Site::REF_MOTO_DE => [CallGroupsMoto::DE_MOTO_CHECKLIST],
            ],
            self::CLASSIFICATION_DEMOLITION_ID => [
                Site::REF_CAR_NL => [CallGroups::NL_CAR_DEMOLITION],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_DEMOLITION],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_DEMOLITION],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_DEMOLITION],
                Site::REF_SCOOTER_NL => [CallGroupsMoto::NL_MOPED_DEMOLITION],
                Site::REF_MOTO_NL => [
                    CallGroupsMoto::NL_MOTO_GROUP_1,
                    CallGroupsMoto::NL_MOTO_GROUP_2,
                    CallGroupsMoto::NL_MOTO_GROUP_3,
                    CallGroupsMoto::NL_MOTO_GROUP_4,
                ],
                Site::REF_MOTO_BE => [
                    CallGroupsMoto::BE_MOTO_GROUP_1,
                    CallGroupsMoto::BE_MOTO_GROUP_2,
                    CallGroupsMoto::BE_MOTO_GROUP_3,
                    CallGroupsMoto::BE_MOTO_GROUP_4,
                ],
                Site::REF_MOTO_DE => [
                    CallGroupsMoto::DE_MOTO_GROUP_1,
                    CallGroupsMoto::DE_MOTO_GROUP_2,
                ],
                Site::REF_BICYCLE_NL => [CallGroupsMoto::NL_BICYCLE_NON_ELECTRIC],
            ],
            self::CLASSIFICATION_EXPORT_ID => [
                Site::REF_CAR_NL => [CallGroups::NL_CAR_EXPORT_1],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_EXPORT],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_EXPORT],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_EXPORT],
            ],
            self::CLASSIFICATION_OTHER_ID => [
                Site::REF_CAR_NL => [
                    CallGroups::NL_CAR_OTHER_1,
                    CallGroups::NL_CAR_OTHER_2,
                ],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_OTHER],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_OTHER],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_OTHER],
                Site::REF_MOTO_NL => [
                    CallGroupsMoto::NL_MOTO_GROUP_1,
                    CallGroupsMoto::NL_MOTO_GROUP_2,
                    CallGroupsMoto::NL_MOTO_GROUP_3,
                    CallGroupsMoto::NL_MOTO_GROUP_4,
                ],
                Site::REF_MOTO_BE => [
                    CallGroupsMoto::BE_MOTO_GROUP_1,
                    CallGroupsMoto::BE_MOTO_GROUP_2,
                    CallGroupsMoto::BE_MOTO_GROUP_3,
                    CallGroupsMoto::BE_MOTO_GROUP_4,
                ],
                Site::REF_MOTO_DE => [
                    CallGroupsMoto::DE_MOTO_GROUP_1,
                    CallGroupsMoto::DE_MOTO_GROUP_2,
                ],
                Site::REF_SCOOTER_NL => [CallGroupsMoto::NL_MOPED_OTHER],
            ],
            self::CLASSIFICATION_YOUNG_ID => [
                Site::REF_CAR_NL => [CallGroups::NL_CAR_YOUNG_FRESH_1],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_YOUNG_FRESH],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_YOUNG_FRESH],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_YOUNG_FRESH],
                Site::REF_SCOOTER_NL => [CallGroupsMoto::NL_MOPED_YOUNG_FRESH],
                Site::REF_MOTO_NL => [
                    CallGroupsMoto::NL_MOTO_GROUP_1,
                    CallGroupsMoto::NL_MOTO_GROUP_2,
                    CallGroupsMoto::NL_MOTO_GROUP_3,
                    CallGroupsMoto::NL_MOTO_GROUP_4
                ],
                Site::REF_MOTO_BE => [
                    CallGroupsMoto::BE_MOTO_GROUP_1,
                    CallGroupsMoto::BE_MOTO_GROUP_2,
                    CallGroupsMoto::BE_MOTO_GROUP_3,
                    CallGroupsMoto::BE_MOTO_GROUP_4
                ],
                Site::REF_MOTO_DE => [
                    CallGroupsMoto::DE_MOTO_GROUP_1,
                    CallGroupsMoto::DE_MOTO_GROUP_2,
                ],
                Site::REF_BICYCLE_NL => [CallGroupsMoto::NL_BICYCLE_ELECTRIC],
            ],
            self::CLASSIFICATION_BUDGET_ID => [
                Site::REF_CAR_NL => [
                    CallGroups::NL_CAR_BUDGET_1,
                    CallGroups::NL_CAR_BUDGET_2,
                ],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_BUDGET],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_BUDGET],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_BUDGET],
                Site::REF_MOTO_NL => [
                    CallGroupsMoto::NL_MOTO_GROUP_1,
                    CallGroupsMoto::NL_MOTO_GROUP_2,
                    CallGroupsMoto::NL_MOTO_GROUP_3,
                    CallGroupsMoto::NL_MOTO_GROUP_4,
                ],
                Site::REF_MOTO_BE => [
                    CallGroupsMoto::BE_MOTO_GROUP_1,
                    CallGroupsMoto::BE_MOTO_GROUP_2,
                    CallGroupsMoto::BE_MOTO_GROUP_3,
                    CallGroupsMoto::BE_MOTO_GROUP_4,
                ],
                Site::REF_MOTO_DE => [
                    CallGroupsMoto::DE_MOTO_GROUP_1,
                    CallGroupsMoto::DE_MOTO_GROUP_2,
                ],
                Site::REF_SCOOTER_NL => [CallGroupsMoto::NL_MOPED_OTHER],
            ],
            self::CLASSIFICATION_PREMIUM_ID => [
                Site::REF_CAR_NL => [CallGroups::NL_CAR_PREMIUM_CLASSIC],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_PREMIUM_CLASSIC],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_PREMIUM_CLASSIC],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_PREMIUM_CLASSIC],
            ],
            self::CLASSIFICATION_RECREATIONAL_ID => [
                Site::REF_CAR_NL => [CallGroups::NL_CAR_RECREATION],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_RECREATION],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_RECREATION],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_RECREATION],
            ],
            self::CLASSIFICATION_TRANSPORT_ID => [
                Site::REF_CAR_NL => [CallGroups::NL_CAR_ORDER_FREIGHT],
                Site::REF_CAR_BE_NL => [CallGroups::BE_CAR_ORDER_FREIGHT],
                Site::REF_CAR_BE_FR => [CallGroups::BEFR_CAR_WALLONIA],
                Site::REF_CAR_DE => [CallGroups::DE_CAR_ORDER_FREIGHT],
                Site::REF_CAR_MARKTPLAATS_NL => [CallGroups::NL_CAR_MP_ORDER_FREIGHT],

                // Hack for now, we don't have a separate group for this
                Site::REF_BICYCLE_NL => [CallGroupsMoto::NL_BICYCLE_B2B],
            ],
        };
    }
}
