<?php

namespace Dealerdirect\VehicleClassification\Rules\Cars;

use Dealerdirect\VehicleClassification\Models\FuelUtils;
use Dealerdirect\VehicleClassification\Models\Vehicle;
use Dealerdirect\VehicleClassification\Models\VehicleTypeCarBodyType;
use Dealerdirect\VehicleClassification\Rules\Operators\CatchAll;
use Dealerdirect\VehicleClassification\Rules\RuleSet as BaseRuleSet;
use Dealerdirect\VehicleClassification\VehicleClassificationIds;
use Ruler\Operator\LogicalOr;

class RuleSet extends BaseRuleSet
{
    public function __construct()
    {
        $rb = $this->getRuleBuilder();
        $currentYear = (int) date('Y');

        $this->addRule(
            $rb->create(
                new LogicalOr(
                    [
                        $rb['vehicleType']->equalTo(Vehicle::TYPE_RECREATIONAL),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::CAMPER),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::CARAVAN),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::FOLDING_CARAVAN),
                    ]
                ),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_RECREATIONAL_ID
            )
        );

        $this->addRule(
            $rb->create(
                new LogicalOr(
                    [
                        $rb['vehicleType']->equalTo(Vehicle::TYPE_TRANSPORT),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::VAN_SINGLE_CABIN),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::VAN_DOUBLE_CABIN),
                        $rb['licensePlate']->companyLicensePlate(),
                    ]
                ),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_TRANSPORT_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->isEmpty(),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_NO_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 30),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_PREMIUM_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 13),
                        $rb['yearOfManufacture']->greaterThanOrEqualTo($currentYear - 24),
                        $rb['mileage']->lessThan(199999),
                    ),
                    $rb->logicalAnd(
                        $rb['makeName']->equalTo('TOYOTA'),
                        $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 20)
                    ),
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 16),
                        $rb['yearOfManufacture']->greaterThanOrEqualTo($currentYear - 18),
                        $rb['mileage']->greaterThanOrEqualTo(200000),
                        $rb['fuelType']->equalTo(FuelUtils::GASOLINE),
                    )
                ),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_BUDGET_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->greaterThanOrEqualTo($currentYear - 15),
                        $rb['mileage']->greaterThan(199999)
                    ),
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->greaterThanOrEqualTo($currentYear - 17),
                        $rb['mileage']->greaterThan(199999),
                        $rb['fuelType']->equalTo(FuelUtils::DIESEL)
                    ),
                ),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_EXPORT_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 19),
                        $rb['mileage']->greaterThanOrEqualTo(200000)
                    ),
                    $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 24),
                ),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_DEMOLITION_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 9),
                    $rb['mileage']->greaterThanOrEqualTo(120000),
                ),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_OTHER_ID
            )
        );

        // Catch all
        $this->addRule(
            $rb->create(
                new CatchAll(),
                fn () => $this->result = VehicleClassificationIds::CLASSIFICATION_YOUNG_ID
            )
        );
    }
}
