<?php

namespace Dealerdirect\VehicleClassification\Tests\Models;

use Dealerdirect\VehicleClassification\Models\CarOption;
use Dealerdirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use Dealerdirect\VehicleClassification\Models\Vehicle;
use Dealerdirect\VehicleClassification\Models\VehicleCondition;
use Dealerdirect\VehicleClassification\Models\VehicleCountry;
use Dealerdirect\VehicleClassification\Utils\ArrayUtils;
use PHPUnit\Framework\TestCase;

class VehicleMapperTest extends TestCase
{
    /**
     * Test vehicle array to Vehicle model

     * @param array $vehicleData The vehicle as array
     *
     * @dataProvider vehicleArraysDataProvider
     */
    public function testVehicleArrayIsCorrectlyMappedToVehicleModel(array $vehicleData)
    {
        $mapper = new VehicleModelMapper();

        $vehicle = $mapper->toVehicle($vehicleData);

        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $this->assertEquals($vehicle->getId(), ArrayUtils::getInt($vehicleData, 'id'));
        $this->assertEquals($vehicle->getVin(), ArrayUtils::getValue($vehicleData, 'vin'));
        $this->assertEquals($vehicle->getLicensePlate(), ArrayUtils::getValue($vehicleData, 'licensePlate'));
        $this->assertEquals($vehicle->getVehicleType(), ArrayUtils::getValue($vehicleData, 'vehicleType'));
        $this->assertEquals($vehicle->getCountry(), ArrayUtils::getInt($vehicleData, 'country'));

        $this->assertEquals($vehicle->isDamageFree(), ArrayUtils::getBoolean($vehicleData, 'damageFree', true));
        $this->assertEquals($vehicle->isDrivable(), ArrayUtils::getBoolean($vehicleData, 'driveable', true));

        $this->assertEquals($vehicle->getMakeName(), ArrayUtils::getValue($vehicleData, 'makeName'));
        $this->assertEquals($vehicle->getModelName(), ArrayUtils::getValue($vehicleData, 'modelName'));
        $this->assertEquals($vehicle->getYearOfManufacture(), ArrayUtils::getInt($vehicleData, 'yearOfManufacture'));
        $this->assertEquals($vehicle->getFuelType(), ArrayUtils::getValue($vehicleData, 'fuelType'));

        $this->assertEquals($vehicle->getMileage(), ArrayUtils::getInt($vehicleData, 'mileage'));
        $this->assertEquals($vehicle->getBpm(), ArrayUtils::getInt($vehicleData, 'bpm'));

        $this->assertEquals($vehicle->getAskingPrice(), ArrayUtils::getInt($vehicleData, 'askingPrice'));
        $this->assertEquals($vehicle->getCatalogPrice(), ArrayUtils::getInt($vehicleData, 'listPrice'));
        $this->assertEquals($vehicle->getValuationPrice(), ArrayUtils::getInt($vehicleData, 'valuationPrice'));
        $this->assertEquals($vehicle->getActualValue(), ArrayUtils::getInt($vehicleData, 'actualValue'));

        $this->assertEquals($vehicle->getOptions(), ArrayUtils::getArray($vehicleData, 'options'));
    }

    /**
     * @param \Dealerdirect\VehicleClassification\Models\Vehicle $vehicle
     *
     * @dataProvider vehicleModelsDataProvider
     */
    public function testVehicleModelIsCorrectlyMappedToVehicleArray(Vehicle $vehicle)
    {
        $mapper = new VehicleModelMapper();

        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleData = $mapper->toArray($vehicle);

        $this->assertEquals($vehicle->getId(), ArrayUtils::getInt($vehicleData, 'id'));
        $this->assertEquals($vehicle->getVin(), ArrayUtils::getValue($vehicleData, 'vin'));
        $this->assertEquals($vehicle->getLicensePlate(), ArrayUtils::getValue($vehicleData, 'licensePlate'));
        $this->assertEquals($vehicle->getVehicleType(), ArrayUtils::getInt($vehicleData, 'vehicleType'));
        $this->assertEquals($vehicle->getCountry(), ArrayUtils::getInt($vehicleData, 'country'));

        $this->assertEquals($vehicle->isDamageFree(), ArrayUtils::getBoolean($vehicleData, 'damageFree', true));
        $this->assertEquals($vehicle->isDrivable(), ArrayUtils::getBoolean($vehicleData, 'drivable', true));

        $this->assertEquals($vehicle->getMakeName(), ArrayUtils::getValue($vehicleData, 'makeName'));
        $this->assertEquals($vehicle->getModelName(), ArrayUtils::getValue($vehicleData, 'modelName'));
        $this->assertEquals($vehicle->getYearOfManufacture(), ArrayUtils::getInt($vehicleData, 'yearOfManufacture'));
        $this->assertEquals($vehicle->getFuelType(), ArrayUtils::getValue($vehicleData, 'fuelType'));

        $this->assertEquals($vehicle->getMileage(), ArrayUtils::getInt($vehicleData, 'mileage'));
        $this->assertEquals($vehicle->getBpm(), ArrayUtils::getInt($vehicleData, 'bpm'));

        $this->assertEquals($vehicle->getAskingPrice(), ArrayUtils::getInt($vehicleData, 'askingPrice'));
        $this->assertEquals($vehicle->getCatalogPrice(), ArrayUtils::getInt($vehicleData, 'listPrice'));
        $this->assertEquals($vehicle->getValuationPrice(), ArrayUtils::getInt($vehicleData, 'valuationPrice'));
        $this->assertEquals($vehicle->getActualValue(), ArrayUtils::getInt($vehicleData, 'actualValue'));

        $this->assertEquals($vehicle->getOptions(), ArrayUtils::getArray($vehicleData, 'options'));
    }

    public function testVehicleMapperSupportsAllAttributesDefinedInVehicleModel()
    {
        $vehicle = new Vehicle();
        $getMethods = [];
        foreach (get_class_methods(Vehicle::class) as $method) {
            if (str_starts_with($method, 'get')) {
                $getMethods[] = $method;
            }

            if (!str_starts_with($method, 'set')) {
                continue;
            }

            $vehicle->$method($method === 'setOptions' ? [] : random_int(123456, 99999999));
        }

        $mapper = new VehicleModelMapper();
        $vehicleArray = $mapper->toArray($vehicle);
        $backToVehicle = $mapper->toVehicle($vehicleArray);

        foreach ($getMethods as $method) {
            $this->assertEquals(
                $vehicle->$method(),
                $backToVehicle->$method(),
                "Failed asserting that '$method' is correctly mapped"
            );
        }
    }

    public static function vehicleModelsDataProvider()
    {
        return array(
            array(
                (new Vehicle())
                    ->setId(1)
                    ->setLicensePlate('9XRX22')
                    ->setCondition(VehicleCondition::GOOD)
                    ->setDrivable(true)
                    ->setDamageFree(true)
                    ->setMakeName('CITROEN')
                    ->setModelName('SAXO')
                    ->setYearOfManufacture(2001)
                    ->setFuelType('benzine')
                    ->setMileage(113249)
                    ->setNumberOfDoors(4)
                    ->setBpm(1200)
                    ->setFuelType(1)
                    ->setOptions(
                        array(
                            1, 2, 3, 4
                        )
                    )
            ),
            array(
                (new Vehicle())
                    ->setId(1)
                    ->setLicensePlate('V32B44')
                    ->setCondition(VehicleCondition::GOOD)
                    ->setCountry(VehicleCountry::NETHERLANDS)
                    ->setDrivable(true)
                    ->setDamageFree(true)
                    ->setMakeName('RENAULT')
                    ->setModelName('KANGOO')
                    ->setYearOfManufacture(2001)
                    ->setFuelType('diesel')
                    ->setMileage(113249)
                    ->setNumberOfDoors(4)
                    ->setBpm(1200)
                    ->setFuelType(1)
                    ->setOptions(
                        array(
                            1, 2, 3, 4
                        )
                    )
            )
        );
    }

    public static function vehicleArraysDataProvider()
    {
        return
            array(
                array(array(
                    'id' => '2840652',
                    'licensePlate' => '',
                    'condition' => VehicleCondition::AVERAGE,
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'country' => VehicleCountry::NETHERLANDS,
                    'damageFree' => '0',
                    'drivable' => '1',
                    'mileage' => '170700',
                    'makeName' => 'FIAT',
                    'modelName' => 'DUCATO',
                    'yearOfManufacture' => '1995',
                    'fuelType' => '2',
                    'bpm' => '0',
                    'askingPrice' => '10000',
                    'listPrice' => 22000,
                    'options' => array(
                        CarOption::AIR_CONDITIONING,
                        CarOption::ABS
                    )
                )) ,
                array(array(
                    'id' => '2840651',
                    'licensePlate' => '',
                    'condition' => VehicleCondition::GOOD,
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'country' => VehicleCountry::BELGIUM,
                    'damageFree' => '0',
                    'drivable' => '1',
                    'mileage' => '246000',
                    'makeName' => 'MAZDA',
                    'modelName' => '6',
                    'yearOfManufacture' => '2006',
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => '3000',
                    'listPrice' => 23300,
                )) ,
                array(array(
                    'id' => '2840650',
                    'licensePlate' => '14DXZR',
                    'condition' => VehicleCondition::POOR,
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '0',
                    'drivable' => '2',
                    'mileage' => '190000',
                    'makeName' => 'HYUNDAI',
                    'modelName' => 'ATOS',
                    'yearOfManufacture' => '2000',
                    'fuelType' => '1',
                    'bpm' => '1016',
                    'askingPrice' => '250',
                    'listPrice' => 11000,
                )) ,
                array(array(
                    'id' => '2840649',
                    'licensePlate' => '61LBKD',
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'condition' => VehicleCondition::BAD,
                    'damageFree' => '0',
                    'drivable' => '1',
                    'mileage' => '132000',
                    'makeName' => 'HYUNDAI',
                    'modelName' => 'ATOS SPIRIT',
                    'yearOfManufacture' => '2002',
                    'fuelType' => '1',
                    'bpm' => '1761',
                    'askingPrice' => '0',
                    'listPrice' => 12000,
                )) ,
                array(array(
                    'id' => '2840648',
                    'licensePlate' => '',
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => '83000',
                    'makeName' => 'KIA',
                    'modelName' => 'C EED',
                    'yearOfManufacture' => '2011',
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => '5600'
                )) ,
                array(array(
                    'id' => '2840647',
                    'licensePlate' => '97XZNV',
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => '194321',
                    'makeName' => 'PEUGEOT',
                    'modelName' => '207',
                    'yearOfManufacture' => '2007',
                    'fuelType' => '1',
                    'bpm' => '6046',
                    'askingPrice' => '0'
                )) ,
                array(array(
                    'id' => '2840646',
                    'licensePlate' => '',
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'condition' => VehicleCondition::AS_NEW,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => '107000',
                    'makeName' => 'CITROEN',
                    'modelName' => 'BERLINGO',
                    'yearOfManufacture' => '2009',
                    'fuelType' => '2',
                    'bpm' => '0',
                    'askingPrice' => '7250'
                ))
            );
    }
}
