<?php

namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

class GroupOtherRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    public function __construct()
    {
        $currentYear = (int)date('Y');

        $this->groupRule = new Rule(
            new Operator\LogicalOr(array(
                // > 30000, 2017 - 2009
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 8),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 30000),
                    ConditionUtils::stringDoesNotContain('makeName', 'HARLEY') // exclude all Harleys of year > 2008
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 14),
                    ConditionUtils::lessThan('mileage', 50000)
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 21),
                    ConditionUtils::lessThan('mileage', 30000)
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    new Operator\LogicalOr(array(
                        ConditionUtils::equalTo('makeName', 'BMW'),
                        ConditionUtils::isOneOf('modelName', ['PAN EUROPEAN', 'FJR']),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    )),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 14),
                    ConditionUtils::lessThan('mileage', 100000)
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', 1980),
                )),
            ))
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_MOTORS_OTHER_ID;
    }
}
