<?php

namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

/**
 * Motor Cycle "Demolition Classification Group" Rules
 *
 * @package DealerDirect\VehicleClassification\Rules\Motors
 */
class GroupDemolitionRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    public function __construct()
    {
        $currentYear = (int)date('Y');

        // Setup exclusions
        $excludedModelsRule = new Operator\LogicalNot([
            new Operator\LogicalOr([
                // Exclude BMW, and special models for mileage, years
                new Operator\LogicalAnd([
                    new Operator\LogicalOr([
                        ConditionUtils::equalTo('makeName', 'BMW'),
                        ConditionUtils::isOneOf('modelName', [
                            'PAN EUROPEAN', 'FJR', 'GOLDWING'
                        ]),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    ]),
                    new Operator\LogicalOr([
                        ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 20),
                    ])
                ]),
                // exclude all Harleys
                ConditionUtils::stringContains('makeName', 'HARLEY'),
                // Exclude classics
                ConditionUtils::equalTo('classic', true),
                ConditionUtils::lessThan('yearOfManufacture', 1980)
            ])
        ]);

        $this->groupRule = new Rule(
            new Operator\LogicalOr(array(
                // Damaged motor cycles
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', false)
                )),
                // Old vehicles, > 60000 km
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 23),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 60000),
                    $excludedModelsRule
                )),
                // Oldest vehicles
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 24),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', 1980),
                    $excludedModelsRule
                )),
            ))
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_MOTORS_DEMOLITION_ID;
    }
}
