<?php

namespace DealerDirect\VehicleClassification\Utils;

use Ruler\Operator;
use Ruler\Set;
use Ruler\Variable;

/**
 * Class Condition
 *
 * Convenience Helper class for Ruler/Rules, to quickly create Variable operators with less mundane code.
 *
 * @package DealerDirect\VehicleClassification\Utils
 */
class ConditionUtils
{

    /**
     * Create a Ruler/Rule 'equalTo' VariableOperator
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\EqualTo
     */
    public static function equalTo($key, $expectedValue)
    {
        return new Operator\EqualTo(
            new Variable($key),
            new Variable($key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'NotEqualTo' VariableOperator
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\NotEqualTo
     */
    public static function notEqualTo($key, $expectedValue)
    {
        return new Operator\NotEqualTo(
            new Variable($key),
            new Variable($key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'greaterThan' VariableOperator
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\GreaterThan
     */
    public static function greaterThan($key, $expectedValue)
    {
        return new Operator\GreaterThan(
            new Variable($key),
            new Variable('min_' . $key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'lessThan' VariableOperator
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\LessThan
     */
    public static function lessThan($key, $expectedValue)
    {
        return new Operator\LessThan(
            new Variable($key),
            new Variable('min_' . $key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'greaterThanOrEqualTo' VariableOperator
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\GreaterThanOrEqualTo
     */
    public static function greaterThanOrEqualTo($key, $expectedValue)
    {
        return new Operator\GreaterThanOrEqualTo(
            new Variable($key),
            new Variable('min_' . $key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'lessThanOrEqualTo' VariableOperator
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\LessThanOrEqualTo
     */
    public static function lessThanOrEqualTo($key, $expectedValue)
    {
        return new Operator\LessThanOrEqualTo(
            new Variable($key),
            new Variable('max_' . $key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'SetContains' VariableOperator
     * where a value should match one of the values in given array
     *
     * @param $key
     * @param array $expectedValues Array containing possible values to be checked against
     *
     * @return Operator\SetContains
     */
    public static function isOneOf($key, array $expectedValues)
    {
        // "Set with [expectedValues] contains 'variable of given key'"
        return new Operator\SetContains(
            new Variable($key . 'Expected', new Set($expectedValues)),
            new Variable($key)
        );
    }

    /**
     * Create a Ruler/Rule 'SetDoesNotContain' VariableOperator
     * where a value should *NOT* match one of the values in given array
     *
     * @param $key
     * @param array $expectedValues Array containing possible values to be checked against
     *
     * @return Operator\SetDoesNotContain
     */
    public static function isNotOneOf($key, array $expectedValues)
    {
        // "Set with [expectedValues] doest NOT contain 'variable of given key'"
        return new Operator\SetDoesNotContain(
            new Variable($key . 'Expected', new Set($expectedValues)),
            new Variable($key)
        );
    }

    /**
     * Create a Ruler/Rule 'StringContains' VariableOperator
     *
     * String check is *case insensitive*!
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\StringContainsInsensitive
     */
    public static function stringContains($key, $expectedValue)
    {
        return new Operator\StringContainsInsensitive(
            new Variable($key),
            new Variable($key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'StringDoesNotContain' VariableOperator
     *
     * String check is *case insensitive*!
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\StringDoesNotContainInsensitive
     */
    public static function stringDoesNotContain($key, $expectedValue)
    {
        return new Operator\StringDoesNotContainInsensitive(
            new Variable($key),
            new Variable($key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'StartsWith' VariableOperator
     *
     * String check is *case insensitive*!
     *
     * @param string $key Variable key name
     * @param mixed $expectedValue
     * @return Operator\StartsWithInsensitive
     */
    public static function startsWith($key, $expectedValue)
    {
        return new Operator\StartsWithInsensitive(
            new Variable($key),
            new Variable($key . 'Expected', $expectedValue)
        );
    }

    /**
     * Create a Ruler/Rule 'ContainsSubset' VariableOperator
     * where an array of values should contain a set of values as given array
     *
     * @param $key
     * @param array $expectedValues Array containing all requred values to be checked against
     *
     * @return Operator\ContainsSubset
     */
    public static function hasValues($key, array $expectedValues)
    {
        // "Set with [expectedValues] contains 'variable of given key'"
        return new Operator\ContainsSubset(
            new Variable($key),
            new Variable($key . 'Expected', new Set($expectedValues))
        );
    }
}
