<?php

namespace Dealerdirect\VehicleClassification\Rules\Cars;

use Dealerdirect\VehicleClassification\Models\Vehicle;
use Dealerdirect\VehicleClassification\Models\VehicleTypeCarBodyType;
use Dealerdirect\VehicleClassification\Rules\Operators\CatchAll;
use Dealerdirect\VehicleClassification\Rules\RuleBuilder;
use Dealerdirect\VehicleClassification\Rules\RuleSet as BaseRuleSet;
use Ruler\Operator\LogicalOr;

class RuleSet extends BaseRuleSet
{
    // Cars group
    public const GROUP_NO_ID = 0;
    public const GROUP_DEMOLITION_ID = 1;
    public const GROUP_EXPORT_ID = 2;
    public const GROUP_HIGH_ID = 3;
    public const GROUP_YOUNG_ID = 4;
    public const GROUP_LOW_ID = 5;
    public const GROUP_PREMIUM_ID = 7;
    public const GROUP_PREMIUM_EXCLUSIVE_ID = 8;
    public const GROUP_PREMIUM_CLASSIC_ID = 9;
    public const GROUP_EXPORT_TOYOTA_ID = 11;
    public const GROUP_RECREATIONAL_ID = 10;
    public const GROUP_TRANSPORT_ID = 12;
    public const GROUP_MARKTPLAATS_ID = 13;

    public function __construct()
    {
        $rb = new RuleBuilder();
        $currentYear = (int)date('Y');

        $this->addRule(
            $rb->create(
                $rb['vehicleType']->equalTo(Vehicle::TYPE_MARKTPLAATS_CAR),
                self::GROUP_MARKTPLAATS_ID
            )
        );

        $this->addRule(
            $rb->create(
                new LogicalOr(
                    [
                        $rb['vehicleType']->equalTo(Vehicle::TYPE_RECREATIONAL),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::CAMPER),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::CARAVAN),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::FOLDING_CARAVAN),
                    ]
                ),
                self::GROUP_RECREATIONAL_ID
            )
        );

        $this->addRule(
            $rb->create(
                new LogicalOr(
                    [
                        $rb['vehicleType']->equalTo(Vehicle::TYPE_TRANSPORT),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::VAN_SINGLE_CABIN),
                        $rb['bodyType']->equalTo(VehicleTypeCarBodyType::VAN_DOUBLE_CABIN),
                        $rb['licensePlate']->companyLicensePlate(),
                    ]
                ),
                self::GROUP_TRANSPORT_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->isEmpty(),
                self::GROUP_NO_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb['makeName']->equalTo('TOYOTA'),
                    $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 13)
                ),
                self::GROUP_EXPORT_TOYOTA_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 30),
                self::GROUP_PREMIUM_CLASSIC_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 20),
                self::GROUP_DEMOLITION_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 13),
                    $rb['mileage']->lessThan(199999)
                ),
                self::GROUP_LOW_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb['yearOfManufacture']->greaterThanOrEqualTo($currentYear - 15),
                    $rb['mileage']->greaterThan(199999)
                ),
                self::GROUP_EXPORT_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 13),
                self::GROUP_LOW_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->lessThanOrEqualTo($currentYear - 9),
                self::GROUP_HIGH_ID
            )
        );

        $this->addRule(
            $rb->create(
                $rb['mileage']->greaterThanOrEqualTo(120000),
                self::GROUP_HIGH_ID
            )
        );

        // Catch all
        $this->addRule(
            $rb->create(
                new CatchAll(),
                self::GROUP_YOUNG_ID
            )
        );
    }
}
