<?php

namespace DealerDirect\VehicleClassification\Rules;

use DealerDirect\VehicleClassification\Models\FuelUtils;
use DealerDirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\Cars\GroupChecklistRules;
use DealerDirect\VehicleClassification\Rules\Cars\GroupLowHangingRules;
use Ruler\Context;

class GroupLowHangingRulesTest extends \PHPUnit_Framework_TestCase
{
    /**
     * Test Checklist group ruleset
     * @param array $vehicleData The vehicle as array
     *
     * @dataProvider provideCorrectDataGroupChecklist
     */
    public function testGroupChecklistCorrectData(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $groupChecklistRules = new GroupLowHangingRules();
        $context = new Context($vehicleData);

        $this->assertTrue($groupChecklistRules->evaluate($context));
    }

    /**
     * Test Checklist group ruleset
     * @param array $vehicleData The vehicle as array
     *
     * @dataProvider provideWrongDataGroupChecklist
     */
    public function testGroupChecklistWrongData(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $groupChecklistRules = new GroupChecklistRules();
        $context = new Context($vehicleData);

        $this->assertFalse($groupChecklistRules->evaluate($context));
    }

    /**
     * @return array<array<array>>
     */
    public static function provideCorrectDataGroupChecklist()
    {
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => true,
                'mileage' => 204000,
                'makeName' => 'AUDI',
                'modelName' => 'A3',
                'yearOfManufacture' => 2002,
                'fuelType' => '1',
                'bpm' => '409',
                'valuationPrice' => 300,
                'numberOfDoors' => 5,
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'makeName' => 'PEUGEOT',
                'modelName' => '206',
                'fuelType' => 'benzine',
                'mileage' => '15000',
                'yearOfManufacture' => 2000,
                'bpm' => '2729',
                'con_carrosserie' => '3',
                'damageFree' => '0',
                'drivable' => true,
                'valuationPrice' => 349,
                'carrosserie' => '1',
                'uitvoering' => '206 Gentry 1.4',
                'motorinhoud' => '1360',
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'TOYOTA',
                'modelName' => 'Corolla',
                'fuelType' => FuelUtils::DIESEL,
                'mileage' => 204000,
                'bpm' => 409,
                'valuationPrice' => 100,
                'numberOfDoors' => 3,
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideWrongDataGroupChecklist()
    {
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => false,
                'mileage' => 155000,
                'makeName' => 'AUDI',
                'modelName' => 'A3',
                'yearOfManufacture' => 2009,
                'fuelType' => '1',
                'bpm' => '409',
                'valuationPrice' => 500,
                'numberOfDoors' => 5
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'makeName' => 'PEUGEOT',
                'modelName' => '206',
                'fuelType' => 'benzine',
                'mileage' => '15000',
                'yearOfManufacture' => 2010,
                'bpm' => '2729',
                'damageFree' => '0',
                'driveable' => false,
                'askingPrice' => 450,
                'valuationPrice' => 1200,
                'carrosserie' => '1',
                'uitvoering' => '206 Gentry 1.4',
                'motorinhoud' => '1360'
            )),
            array(array(
                'id' => '1',
                'licensePlate' => '44NJTB',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 316000,
                'makeName' => 'SAAB',
                'modelName' => '9-3',
                'yearOfManufacture' => 2003,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => '10156',
                'numberOfDoors' => 5,
                'askingPrice' => '1650'
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'TOYOTA',
                'modelName' => 'Corolla',
                'fuelType' => FuelUtils::DIESEL,
                'mileage' => 204000,
                'bpm' => 409,
                'valuationPrice' => 100,
                'numberOfDoors' => 3,
                'yearOfManufacture' => 2003,
            )),
        );
    }
}
