<?php

namespace DealerDirect\VehicleClassification\Rules\Cars;

use DealerDirect\VehicleClassification\Models\FuelUtils;
use DealerDirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Models\VehicleTypeCarBodyType;
use DealerDirect\VehicleClassification\Rules\Cars\GroupRecreationalRules;
use Ruler\Context;

class GroupRecreationalRulesTest extends \PHPUnit_Framework_TestCase
{
    /**
     * Test Export group ruleset
     *
     * @param array $vehicleData The vehicle as array
     *
     * @dataProvider provideCorrectDataGroupExportToyota
     */
    public function testGroupRecreationalCorrectData(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $groupRecreationalRules = new GroupRecreationalRules();
        $context = new Context($vehicleData);

        $this->assertTrue($groupRecreationalRules->evaluate($context));
    }

    /**
     * Test young group ruleset
     *
     * @param array $vehicleData The vehicle as array
     *
     * @dataProvider provideWrongDataGroupExportToyota
     */
    public function testGroupRecreationalWrongData(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $groupRecreationalRules = new GroupRecreationalRules();
        $context = new Context($vehicleData);

        $this->assertFalse($groupRecreationalRules->evaluate($context));
    }

    /**
     * @return array<array<array>>
     */
    public static function provideCorrectDataGroupExportToyota()
    {
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_RECREATIONAL,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'FENDT',
                'modelName' => '130',
                'yearOfManufacture' => 1991,
                'mileage' => 204000,
                'bpm' => 409,
                'askingPrice' => 1000,
                'numberOfDoors' => 4
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_RECREATIONAL,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'KIP',
                'modelName' => 'Corolla',
                'yearOfManufacture' => 1992,
                'mileage' => 204000,
                'bpm' => 409,
                'askingPrice' => 1000,
                'numberOfDoors' => 3
            )),
            array(array(
                'bodyType' => VehicleTypeCarBodyType::CARAVAN,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'FRANKIA',
                'modelName' => 'ANDERS',
                'yearOfManufacture' => 2001,
                'askingPrice' => 1000,
            )),
            array(array(
                'bodyType' => VehicleTypeCarBodyType::FOLDING_CARAVAN,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'PENNINE',
                'modelName' => 'CRUISER',
                'yearOfManufacture' => 2011,
                'askingPrice' => 8500,
            )),
            array(array(
                'bodyType' => VehicleTypeCarBodyType::CAMPER,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'HYMER',
                'modelName' => 'B 588',
                'yearOfManufacture' => 2016,
                'askingPrice' => 88000,
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideWrongDataGroupExportToyota()
    {
        return array(
            // Car of 2000, too new
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'TOYOTA',
                'modelName' => 'Corolla',
                'yearOfManufacture' => 2000,
                'fuelType' => FuelUtils::DIESEL,
                'mileage' => 204000,
                'bpm' => 409,
                'askingPrice' => 1000,
                'numberOfDoors' => 5
            )),
            // Car of 1989, too old
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'TOYOTA',
                'modelName' => 'Corolla',
                'yearOfManufacture' => 1989,
                'fuelType' => FuelUtils::GASOLINE,
                'mileage' => 204000,
                'bpm' => 409,
                'askingPrice' => 1000,
                'numberOfDoors' => 5
            )),
            // 2 Doors case, should be excluded
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'TOYOTA',
                'modelName' => 'Celica',
                'yearOfManufacture' => 1993,
                'fuelType' => FuelUtils::GASOLINE,
                'mileage' => 204000,
                'bpm' => 409,
                'askingPrice' => 3000,
                'numberOfDoors' => 2
            )),
            array(array(
                'bodyType' => VehicleTypeCarBodyType::CROSS_COUNTRY,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'VOLVO',
                'modelName' => 'X40',
                'yearOfManufacture' => 2011,
                'askingPrice' => 8500,
            )),
        );
    }
}
