<?php

namespace Dealerdirect\VehicleClassification;

use DealerDirect\Generic\Category\CallGroups;
use DealerDirect\Generic\Category\CallGroupsMoto;
use DealerDirect\Generic\Category\Site;
use Exception;

/**
 * @codeCoverageIgnore
 */
enum VehicleClassificationIds: int
{
    /**
     * Vehicle Classification Id's
     *
     * Values are mapped to the id's in the database table:
     * dealer01_ddmain.vehicle_classification_descriptions
     */
    case CLASSIFICATION_NO_ID = 0;

    case CLASSIFICATION_YOUNG_ID = 1;
    case CLASSIFICATION_DEMOLITION_ID = 11;
    case CLASSIFICATION_RECREATIONAL_ID = 21;
    case CLASSIFICATION_TRANSPORT_ID = 31;
    case CLASSIFICATION_BUDGET_ID = 41;
    case CLASSIFICATION_EXPORT_ID = 51;
    case CLASSIFICATION_OTHER_ID = 61;
    case CLASSIFICATION_PREMIUM_ID = 71;
    case CLASSIFICATION_LOW_POTENTIAL = 72;

    public function toCallGroupId(int $reference, bool $returnAll = false): array|int
    {
        $callGroupIds = match ($reference) {
            Site::REF_CAR_NL => $this->carNlGroups(),
            Site::REF_CAR_BE_NL => $this->carBeGroups(),
            Site::REF_CAR_MARKTPLAATS_NL => $this->carMarktplaatsNlGroups(),
            Site::REF_MOTO_NL => $this->motoNlGroups(),
            Site::REF_MOTO_BE => $this->motoBeGroups(),
            Site::REF_MOTO_DE => $this->motoDeGroups(),

            Site::REF_SCOOTER_NL => $this->scooterNlGroups(),
            Site::REF_BICYCLE_NL => $this->bicycleNlGroups(),

            default => throw new Exception("Error, reference $reference not found in mapping"),
        };

        return $returnAll
            ? $callGroupIds
            : (int) $callGroupIds[array_rand($callGroupIds)];
    }

    private function bicycleNlGroups(): array
    {
        return match ($this) {
            self::CLASSIFICATION_DEMOLITION_ID => [CallGroupsMoto::NL_BICYCLE_NON_ELECTRIC],
            self::CLASSIFICATION_YOUNG_ID => [CallGroupsMoto::NL_BICYCLE_ELECTRIC],
            self::CLASSIFICATION_TRANSPORT_ID => [CallGroupsMoto::NL_BICYCLE_B2B],

            default => throw new Exception("No groups found for bicycle NL classification id $this->value"),
        };
    }

    private function scooterNlGroups(): array
    {
        return match ($this) {
            self::CLASSIFICATION_DEMOLITION_ID => [CallGroupsMoto::NL_MOPED_DEMOLITION],
            self::CLASSIFICATION_YOUNG_ID => [CallGroupsMoto::NL_MOPED_YOUNG_FRESH],
            self::CLASSIFICATION_LOW_POTENTIAL => [CallGroupsMoto::NL_MOPED_LOW_POTENTIAL],
            self::CLASSIFICATION_OTHER_ID,
            self::CLASSIFICATION_BUDGET_ID => [CallGroupsMoto::NL_MOPED_OTHER],

            default => throw new Exception("No groups found for scooter NL classification id $this->value"),
        };
    }

    private function carNlGroups(): array
    {
        return [
            CallGroups::NL_CAR_GROUP_1,
            CallGroups::NL_CAR_GROUP_2,
            CallGroups::NL_CAR_GROUP_3,
            CallGroups::NL_CAR_GROUP_4,
        ];
    }

    private function carBeGroups(): array
    {
        return [
            CallGroups::BE_CAR_GROUP_1,
            CallGroups::BE_CAR_GROUP_2,
            CallGroups::BE_CAR_GROUP_3,
            CallGroups::BE_CAR_GROUP_4,
        ];
    }

    private function carMarktplaatsNlGroups(): array
    {
        return [
            CallGroups::NL_CAR_MARKTPLAATS_GROUP_1,
        ];
    }

    private function motoNlGroups(): array
    {
        return [
            CallGroupsMoto::NL_MOTO_GROUP_1,
            CallGroupsMoto::NL_MOTO_GROUP_2,
            CallGroupsMoto::NL_MOTO_GROUP_3,
            CallGroupsMoto::NL_MOTO_GROUP_4,
        ];
    }

    private function motoBeGroups(): array
    {
        return [
            CallGroupsMoto::BE_MOTO_GROUP_1,
            CallGroupsMoto::BE_MOTO_GROUP_2,
            CallGroupsMoto::BE_MOTO_GROUP_3,
            CallGroupsMoto::BE_MOTO_GROUP_4,
        ];
    }

    private function motoDeGroups(): array
    {
        return [
            CallGroupsMoto::DE_MOTO_GROUP_1,
            CallGroupsMoto::DE_MOTO_GROUP_2,
        ];
    }
}
