<?php

namespace Dealerdirect\VehicleClassification\Tests\Rules\Cars;

use Dealerdirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use Dealerdirect\VehicleClassification\Models\Vehicle;
use Dealerdirect\VehicleClassification\Models\VehicleTypeCarBodyType;
use Dealerdirect\VehicleClassification\Rules\Cars\RuleSet;
use Dealerdirect\VehicleClassification\Models\FuelUtils;
use Dealerdirect\VehicleClassification\Models\CarOption;
use Dealerdirect\VehicleClassification\Models\VehicleCountry;
use PHPUnit\Framework\TestCase;
use Ruler\Context;

class RuleSetTest extends TestCase
{
    /**
     * @var RuleSet
     */
    private $ruleSet;

    protected function setUp(): void
    {
        $this->ruleSet = new RuleSet();
    }


    /**
     * Test cars ruleset
     *
     * @param array $vehicleData The vehicle as array
     * @param int $expectedGroupId
     *
     * @dataProvider provideExpectedData
     */
    public function testRuleSetShouldClassifySubjectsInExpectedGroup(array $vehicleData, $expectedGroupId)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $context = new Context($vehicleData);


        $this->assertEquals($expectedGroupId, $this->ruleSet->executeRules($context));
    }


    /**
     * @return array<array<array>>
     */
    public static function provideExpectedData()
    {
        $currentYear = (int)date('Y');
        return [
            // Checklist
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 204000,
                    'makeName' => 'AUDI',
                    'modelName' => 'A3',
                    'yearOfManufacture' => 0,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => 200,
                    'numberOfDoors' => 5
                ],
                RuleSet::GROUP_NO_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'makeName' => 'PEUGEOT',
                    'modelName' => '206',
                    'fuelType' => 'benzine',
                    'mileage' => '15000',
                    'yearOfManufacture' => '',
                    'bpm' => '2729',
                    'con_carrosserie' => '3',
                    'damageFree' => '0',
                    'driveable' => '1',
                    'askingPrice' => 450,
                    'actualValue' => 250,
                    'carrosserie' => '1',
                    'uitvoering' => '206 Gentry 1.4',
                    'motorinhoud' => '1360'
                ],
                RuleSet::GROUP_NO_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => false,
                    'makeName' => 'TOYOTA',
                    'modelName' => 'Corolla',
                    'fuelType' => FuelUtils::DIESEL,
                    'mileage' => 204000,
                    'bpm' => 409,
                    'askingPrice' => 1000,
                    'numberOfDoors' => 3
                ],
                RuleSet::GROUP_NO_ID
            ],
            // Demolition
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 204000,
                    'makeName' => 'AUDI',
                    'modelName' => 'A3',
                    'yearOfManufacture' => $currentYear - 22,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => 200,
                    'numberOfDoors' => 5
                ],
                RuleSet::GROUP_DEMOLITION_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'makeName' => 'PEUGEOT',
                    'modelName' => '206',
                    'fuelType' => 'benzine',
                    'mileage' => '15000',
                    'yearOfManufacture' => $currentYear - 21,
                    'bpm' => '2729',
                    'con_carrosserie' => '3',
                    'damageFree' => '0',
                    'driveable' => '1',
                    'askingPrice' => 450,
                    'actualValue' => 250,
                    'carrosserie' => '1',
                    'uitvoering' => '206 Gentry 1.4',
                    'motorinhoud' => '1360'
                ],
                RuleSet::GROUP_DEMOLITION_ID
            ],
            // Export
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 204000,
                    'makeName' => 'AUDI',
                    'modelName' => 'A3',
                    'yearOfManufacture' => $currentYear - 15,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => 30000,
                    'numberOfDoors' => 5,
                    'options' => [
                        CarOption::AIR_CONDITIONING,
                        CarOption::ABS,
                        CarOption::POWER_STEERING,
                        CarOption::ALLOY_WHEELS,
                        CarOption::CRUISE_CONTROL
                    ]
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 230000,
                    'makeName' => 'SKODA',
                    'modelName' => 'SUPERB',
                    'yearOfManufacture' => $currentYear - 11,
                    'fuelType' => '1',
                    'bpm' => '8284',
                    'askingPrice' => 17000,
                    'numberOfDoors' => 3
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 232000,
                    'makeName' => 'SKODA',
                    'modelName' => 'SUPERB - 2 DOOR COUPE',
                    'yearOfManufacture' => $currentYear - 11,
                    'fuelType' => '1',
                    'bpm' => '8284',
                    'askingPrice' => 17000,
                    'numberOfDoors' => 2
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 232000,
                    'makeName' => 'SKODA',
                    'modelName' => 'SUPERB - 4 DOOR SEDAN',
                    'yearOfManufacture' => $currentYear - 11,
                    'fuelType' => '1',
                    'bpm' => '8284',
                    'askingPrice' => 17000,
                    'numberOfDoors' => 4
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 232000,
                    'makeName' => 'SKODA',
                    'modelName' => 'SUPERB - 5 DOOR STATION',
                    'yearOfManufacture' => $currentYear - 11,
                    'fuelType' => '1',
                    'bpm' => '8284',
                    'askingPrice' => 17000,
                    'numberOfDoors' => 5
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => 0,
                    'drivable' => 1,
                    'mileage' => '200001',
                    'makeName' => 'CHEVROLET',
                    'modelName' => 'MATIZ',
                    'yearOfManufacture' => $currentYear - 13,
                    'fuelType' => FuelUtils::GASOLINE,
                    'bpm' => 1968,
                    'askingPrice' => 1850
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => false,
                    'mileage' => '200000',
                    'makeName' => 'SEAT',
                    'modelName' => 'IBIZA',
                    'yearOfManufacture' => $currentYear - 14,
                    'fuelType' => FuelUtils::GASOLINE,
                    'bpm' => 3024,
                    'askingPrice' => 0
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 204000,
                    'makeName' => 'AUDI',
                    'modelName' => 'A3 - 4 DOOR SEDAN',
                    'yearOfManufacture' => $currentYear - 14,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => 29000,
                    'numberOfDoors' => 4,
                    'options' => [
                        CarOption::AIR_CONDITIONING,
                        CarOption::ABS,
                        CarOption::POWER_STEERING,
                        CarOption::ALLOY_WHEELS,
                        CarOption::CRUISE_CONTROL
                    ],
                ],
                RuleSet::GROUP_EXPORT_ID
            ],
            // High
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 120000,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 9,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_HIGH_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 90000,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 10,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_HIGH_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 120001,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 5,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_HIGH_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 199998,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 12,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_HIGH_ID
            ],
            // Low
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 110000,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 14,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'numberOfDoors' => 3,
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_LOW_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 90000,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 15,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'numberOfDoors' => 4,
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_LOW_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 110000,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 13,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'numberOfDoors' => 3,
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_LOW_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 90000,
                    'makeName' => 'SAAB',
                    'modelName' => '9-3',
                    'yearOfManufacture' => $currentYear - 15,
                    'fuelType' => '1',
                    'bpm' => '10156',
                    'numberOfDoors' => 4,
                    'askingPrice' => 1650
                ],
                RuleSet::GROUP_LOW_ID
            ],
            // Premium Classic
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 204000,
                    'makeName' => 'FERRARI',
                    'modelName' => '488 GTB',
                    'yearOfManufacture' => $currentYear - 31,
                    'fuelType' => FuelUtils::GASOLINE,
                    'bpm' => 11000,
                    'askingPrice' => 200000,
                    'listPrice' => 249150,
                    'numberOfDoors' => 2
                ],
                RuleSet::GROUP_PREMIUM_CLASSIC_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 73000,
                    'makeName' => 'PORSCHE',
                    'modelName' => 'CARRERA',
                    'yearOfManufacture' => $currentYear - 31,
                    'fuelType' => FuelUtils::GASOLINE,
                    'bpm' => 8000,
                    'askingPrice' => 65000,
                    'listPrice' => 89000,
                    'numberOfDoors' => 2
                ],
                RuleSet::GROUP_PREMIUM_CLASSIC_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 56500,
                    'makeName' => 'PORSCHE',
                    'modelName' => '911 TURBO',
                    'yearOfManufacture' => $currentYear - 32,
                    'fuelType' => FuelUtils::GASOLINE,
                    'bpm' => 11000,
                    'askingPrice' => 80000,
                    'listPrice' => 120000,
                    'numberOfDoors' => 2
                ],
                RuleSet::GROUP_PREMIUM_CLASSIC_ID
            ],
            // Recreation Vehicle
            [
                [
                    'vehicleType' => Vehicle::TYPE_RECREATIONAL,
                    'damageFree' => true,
                    'drivable' => true,
                    'makeName' => 'FENDT',
                    'modelName' => '130',
                    'yearOfManufacture' => $currentYear - 28,
                    'mileage' => 204000,
                    'bpm' => 409,
                    'askingPrice' => 1000,
                    'numberOfDoors' => 4
                ],
                RuleSet::GROUP_RECREATIONAL_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_RECREATIONAL,
                    'damageFree' => true,
                    'drivable' => true,
                    'makeName' => 'KIP',
                    'modelName' => 'Corolla',
                    'yearOfManufacture' => $currentYear - 27,
                    'mileage' => 204000,
                    'bpm' => 409,
                    'askingPrice' => 1000,
                    'numberOfDoors' => 3
                ],
                RuleSet::GROUP_RECREATIONAL_ID
            ],
            [
                [
                    'bodyType' => VehicleTypeCarBodyType::CARAVAN,
                    'damageFree' => true,
                    'drivable' => true,
                    'makeName' => 'FRANKIA',
                    'modelName' => 'ANDERS',
                    'yearOfManufacture' => $currentYear - 19,
                    'askingPrice' => 1000,
                ],
                RuleSet::GROUP_RECREATIONAL_ID
            ],
            [
                [
                    'bodyType' => VehicleTypeCarBodyType::FOLDING_CARAVAN,
                    'damageFree' => true,
                    'drivable' => true,
                    'makeName' => 'PENNINE',
                    'modelName' => 'CRUISER',
                    'yearOfManufacture' => $currentYear - 18,
                    'askingPrice' => 8500,
                ],
                RuleSet::GROUP_RECREATIONAL_ID
            ],
            [
                [
                    'bodyType' => VehicleTypeCarBodyType::CAMPER,
                    'damageFree' => true,
                    'drivable' => true,
                    'makeName' => 'HYMER',
                    'modelName' => 'B 588',
                    'yearOfManufacture' => $currentYear - 5,
                    'askingPrice' => 88000,
                ],
                RuleSet::GROUP_RECREATIONAL_ID
            ],
            // Toyota Export
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'makeName' => 'TOYOTA',
                    'modelName' => 'Corolla',
                    'yearOfManufacture' => $currentYear - 26,
                    'fuelType' => FuelUtils::GASOLINE,
                    'mileage' => 204000,
                    'bpm' => 409,
                    'askingPrice' => 1000,
                    'numberOfDoors' => 4
                ],
                RuleSet::GROUP_EXPORT_TOYOTA_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'makeName' => 'TOYOTA',
                    'modelName' => 'Corolla',
                    'yearOfManufacture' => $currentYear - 16,
                    'fuelType' => FuelUtils::DIESEL,
                    'mileage' => 204000,
                    'bpm' => 409,
                    'askingPrice' => 1000,
                    'numberOfDoors' => 3
                ],
                RuleSet::GROUP_EXPORT_TOYOTA_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => false,
                    'makeName' => 'TOYOTA',
                    'modelName' => 'Corolla',
                    'yearOfManufacture' => ($currentYear - 15),
                    'fuelType' => FuelUtils::DIESEL,
                    'mileage' => 204000,
                    'bpm' => 409,
                    'askingPrice' => 1000,
                    'numberOfDoors' => 3
                ],
                RuleSet::GROUP_EXPORT_TOYOTA_ID
            ],
            // Transport
            [
                [
                    'vehicleType' => Vehicle::TYPE_TRANSPORT,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'licensePlate' => 'VJ452B',
                    'mileage' => 60000,
                    'makeName' => 'RENAULT',
                    'modelName' => 'KANGOO',
                    'yearOfManufacture' => $currentYear - 4,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => '30000'
                ],
                RuleSet::GROUP_TRANSPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_TRANSPORT,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'licensePlate' => 'XJ56AB',
                    'mileage' => 60000,
                    'makeName' => 'RENAULT',
                    'modelName' => 'KANGOO',
                    'yearOfManufacture' => $currentYear - 4,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => '30000'
                ],
                RuleSet::GROUP_TRANSPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_UNKNOWN,
                    'country' => VehicleCountry::NETHERLANDS,
                    'licensePlate' => 'V32B44',
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 65000,
                    'makeName' => 'OPEL',
                    'modelName' => 'VIVARO',
                    'yearOfManufacture' => $currentYear - 4,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => '30000'
                ],
                RuleSet::GROUP_TRANSPORT_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'country' => VehicleCountry::NETHERLANDS,
                    'licensePlate' => 'B37AB1',
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 141954,
                    'makeName' => 'VOLKSWAGEN',
                    'modelName' => 'TRANSPORTER',
                    'yearOfManufacture' => $currentYear - 20,
                    'fuelType' => '1',
                    'bpm' => '2434',
                    'askingPrice' => '1000'
                ],
                RuleSet::GROUP_TRANSPORT_ID
            ],
            [
                [
                    'country' => VehicleCountry::BELGIUM,
                    'bodyType' => VehicleTypeCarBodyType::VAN_SINGLE_CABIN,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 30000,
                    'makeName' => 'RENAULT',
                    'modelName' => 'KANGOO',
                    'yearOfManufacture' => $currentYear - 4,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => '30000'
                ],
                RuleSet::GROUP_TRANSPORT_ID
            ],
            [
                [
                    'country' => VehicleCountry::BELGIUM,
                    'bodyType' => VehicleTypeCarBodyType::VAN_DOUBLE_CABIN,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'licensePlate' => 'VJ452B',
                    'mileage' => 60000,
                    'makeName' => 'MERCEDES',
                    'modelName' => 'SPRINTER',
                    'yearOfManufacture' => $currentYear - 3,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => '30000'
                ],
                RuleSet::GROUP_TRANSPORT_ID
            ],
            // Young
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 88000,
                    'makeName' => 'VOLVO',
                    'modelName' => 'AMAZON',
                    'yearOfManufacture' => $currentYear - 8,
                    'fuelType' => '1',
                    'bpm' => '0',
                    'askingPrice' => '0'
                ],
                RuleSet::GROUP_YOUNG_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => true,
                    'drivable' => true,
                    'mileage' => 119999,
                    'makeName' => 'SKODA',
                    'modelName' => 'SUPERB',
                    'yearOfManufacture' => $currentYear - 7,
                    'fuelType' => '1',
                    'bpm' => '8284',
                    'askingPrice' => '17000'
                ],
                RuleSet::GROUP_YOUNG_ID
            ],
            [
                [
                    'vehicleType' => Vehicle::TYPE_CAR,
                    'damageFree' => '1',
                    'drivable' => '1',
                    'mileage' => 60000,
                    'makeName' => 'AUDI',
                    'modelName' => 'A3',
                    'yearOfManufacture' => $currentYear - 4,
                    'fuelType' => '1',
                    'bpm' => '409',
                    'askingPrice' => '30000'
                ],
                RuleSet::GROUP_YOUNG_ID
            ],
        ];
    }
}
