<?php
namespace DealerDirect\VehicleClassification;

use DealerDirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\BaseCarRules;
use DealerDirect\VehicleClassification\Rules\BaseMotorRules;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Rules\VehicleCarRules;
use Ruler\Context;

/**
 * Class VehicleClassification
 *
 * Main Vehicle Classification processor.
 *
 * @package DealerDirect\VehicleClassification
 */
class VehicleClassification
{

    /**
     * @var array
     */
    private $groupRules;

    /**
     * @var VehicleModelMapper
     */
    private $vehicleMapper;

    /**
     * VehicleClassification constructor.
     *
     * @param array $groupRulesSet A set of group rules
     */
    protected function __construct(array $groupRulesSet = array()) {
        $this->vehicleMapper = new VehicleModelMapper();
        $this->groupRules = $groupRulesSet;
    }

    /**
     * Create a VehicleClassification instance
     *
     * @param array $groupRulesSet A set of group rules
     *
     * @return VehicleClassification
     */
    public static function create(array $groupRulesSet = array()) {
        return new VehicleClassification($groupRulesSet);
    }

    /**
     * @return array
     */
    public function getGroupRules()
    {
        return $this->groupRules;
    }

    /**
     * @param array $groupRules
     */
    public function setGroupRules(array $groupRules)
    {
        $this->groupRules = $groupRules;
    }

    /**
     * Classifies a vehicle to a given set of group rules.
     * It will return the Classification Group ID of the first matched group in the
     * Vehicle Classification GroupRules set.
     *
     * @param Vehicle $vehicle
     *
     * @return int Group ID in which vehicle will be classified
     */
    public function classifyVehicle(Vehicle $vehicle) {

        $vehicleContext = new Context(
            $this->vehicleMapper->toArray($vehicle)
        );

        /**
         * @var GroupRulesInterface $group
         */
        foreach ($this->groupRules as $group) {
            if ($group->evaluate($vehicleContext)) {
                return $group->getGroupId();
            }
        }

        // Vehicle could not be classified
        return GroupRulesInterface::GROUP_NO_ID;
    }

    /**
     * @param Vehicle $vehicle
     * @return array
     */
    public function suggestClassifications(Vehicle $vehicle) {
        $suggestedGroups = array();

        $vehicleContext = new Context(
            $this->vehicleMapper->toArray($vehicle)
        );

        /**
         * @var GroupRulesInterface $group
         */
        foreach ($this->groupRules as $group) {
            if ($group->evaluate($vehicleContext)) {
                $suggestedGroups[] = $group->getGroupId();
            }
        }

        if (empty($suggestedGroups)) {
            // In case of empty array, add "NO GROUP":
            $suggestedGroups[] = GroupRulesInterface::GROUP_NO_ID;
        }

        return $suggestedGroups;
    }
}
