<?php

namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

/**
 * Motor Cycle "Budget Classification Group" Rules
 *
 * @package DealerDirect\VehicleClassification\Rules\Motors
 */
class GroupBudgetRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    public function __construct()
    {
        $currentYear = (int)date('Y');

        $this->groupRule = new Rule(
            new Operator\LogicalOr(array(
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 50000),
                    ConditionUtils::lessThan('mileage', 80000)
                )),
                // Exceptions: BMW, Pan European, FJR, Goldwing, Valkyrie
                // only with mileage < 100K
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    new Operator\LogicalOr(array(
                        ConditionUtils::equalTo('makeName', 'BMW'),
                        ConditionUtils::isOneOf('modelName', [
                            'PAN EUROPEAN', 'FJR', 'GOLDWING'
                        ]),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    )),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 22),
                    ConditionUtils::lessThan('mileage', 100000)
                ))
            ))
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_MOTORS_BUDGET_ID;
    }
}

