<?php

namespace Dealerdirect\VehicleClassification\Rules\Motors;

use Dealerdirect\VehicleClassification\Models\Vehicle;
use Dealerdirect\VehicleClassification\Models\VehicleTypeMotorBodyType as BodyType;
use Dealerdirect\VehicleClassification\Rules\Operators\CatchAll;
use Dealerdirect\VehicleClassification\Rules\RuleBuilder;
use Dealerdirect\VehicleClassification\Rules\RuleSet as BaseRuleSet;
use Ruler\Operator\LogicalAnd;
use Ruler\Operator\LogicalOr;

class RuleSet extends BaseRuleSet
{

    public const GROUP_NO_ID = 0;

    // Motors group
    public const GROUP_YOUNG_ID = 40;
    public const GROUP_OTHER_ID = 41;
    public const GROUP_DEMOLITION_ID = 42;
    public const GROUP_BUDGET_ID = 43;
    public const GROUP_SCOOTER_OTHER_ID = 44;
    public const GROUP_SCOOTER_YOUNG_ID = 45;
    public const GROUP_SCOOTER_DEMOLITION_ID = 46;

    public function __construct()
    {
        $rb = new RuleBuilder();
        $currentYear = (int)date('Y');

        $this->addRule(
            $rb->create(
                new LogicalOr(
                    [
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Suzuki'),
                            $rb['modelName']->stringContainsInsensitive('VS 1400 INTRUDER')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Honda'),
                            $rb['modelName']->stringContainsInsensitive('XL 1000 V VARADERO')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Honda'),
                            $rb['modelName']->stringContainsInsensitive('STX 1300 PAN EUROPEAN')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Honda'),
                            $rb['modelName']->stringContainsInsensitive('XRV 750 AFRICA TWIN')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Honda'),
                            $rb['modelName']->stringContainsInsensitive('NTV700V DEAUVILLE')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Yamaha'),
                            $rb['modelName']->stringContainsInsensitive('XV 1600 WILDSTAR')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Yamaha'),
                            $rb['modelName']->stringContainsInsensitive('TDM 900')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Yamaha'),
                            $rb['modelName']->stringContainsInsensitive('FJR 1300')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Yamaha'),
                            $rb['modelName']->stringContainsInsensitive('FJR 1300 A')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Yamaha'),
                            $rb['modelName']->stringContainsInsensitive('FJR 1300 AE')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Yamaha'),
                            $rb['modelName']->stringContainsInsensitive('FJR 1300 AS')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Ducati'),
                            $rb['modelName']->stringContainsInsensitive('916')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Ducati'),
                            $rb['modelName']->stringContainsInsensitive('888')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Ducati'),
                            $rb['modelName']->stringContainsInsensitive('888 SP')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Ducati'),
                            $rb['modelName']->stringContainsInsensitive('851')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Ducati'),
                            $rb['modelName']->stringContainsInsensitive('900 SS')
                        ),
                        $rb->logicalAnd(
                            $rb['makeName']->stringContainsInsensitive('Ducati'),
                            $rb['modelName']->stringContainsInsensitive('900 SS SUPERLIGHT')
                        ),
                    ]
                ),
                self::GROUP_BUDGET_ID
            )
        );

        // Type scooter of tot 50cc, Vespa als merk of in het model,
        // jonger dan 5 jaar en minder dan 20.000km
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb->logicalOr(
                            $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                            $rb->logicalAnd(
                                $rb['engineCapacity']->greaterThanOrEqualTo(1),
                                $rb['engineCapacity']->lessThan(50)
                            )
                        ),
                        $rb->logicalOr(
                            $rb['makeName']->stringContainsInsensitive('VESPA'),
                            $rb['modelName']->stringContainsInsensitive('VESPA')
                        ),
                        $rb['mileage']->lessThan(20000),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 5)
                    ]
                ),
                self::GROUP_SCOOTER_YOUNG_ID
            )
        );

        // Type scooter of tot 50cc, Vespa als merk of in het model
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb->logicalOr(
                        $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                        $rb->logicalAnd(
                            $rb['engineCapacity']->greaterThanOrEqualTo(1),
                            $rb['engineCapacity']->lessThan(50)
                        )
                    ),
                    $rb->logicalOr(
                        $rb['makeName']->stringContainsInsensitive('VESPA'),
                        $rb['modelName']->stringContainsInsensitive('VESPA')
                    )
                ),
                self::GROUP_SCOOTER_OTHER_ID
            )
        );

        // Type scooter of tot 50cc, ouder dan 25 jaar
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb->logicalOr(
                        $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                        $rb->logicalAnd(
                            $rb['engineCapacity']->greaterThanOrEqualTo(1),
                            $rb['engineCapacity']->lessThan(50)
                        )
                    ),
                    $rb['yearOfManufacture']->lessThan($currentYear - 25)
                ),
                self::GROUP_SCOOTER_OTHER_ID
            )
        );

        // Type scooter of tot 50cc, meer dan 10.000, jonger dan 5 jaar
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb->logicalOr(
                            $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                            $rb->logicalAnd(
                                $rb['engineCapacity']->greaterThanOrEqualTo(1),
                                $rb['engineCapacity']->lessThan(50)
                            )
                        ),
                        $rb['mileage']->greaterThan(10000),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 5)
                    ]
                ),
                self::GROUP_SCOOTER_OTHER_ID
            )
        );

        // Type scooter of tot 50cc, jonger dan 5 jaar
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb->logicalOr(
                            $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                            $rb->logicalAnd(
                                $rb['engineCapacity']->greaterThanOrEqualTo(1),
                                $rb['engineCapacity']->lessThan(50)
                            )
                        ),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 5)
                    ]
                ),
                self::GROUP_SCOOTER_YOUNG_ID
            )
        );

        // Type scooter of tot 50cc, meer dan 25.000, jonger dan 8 jaar
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb->logicalOr(
                            $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                            $rb->logicalAnd(
                                $rb['engineCapacity']->greaterThanOrEqualTo(1),
                                $rb['engineCapacity']->lessThan(50)
                            )
                        ),
                        $rb['mileage']->greaterThan(25000),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 8)
                    ]
                ),
                self::GROUP_SCOOTER_DEMOLITION_ID
            )
        );

        // Type scooter of tot 50cc, jonger dan 8 jaar
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb->logicalOr(
                            $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                            $rb->logicalAnd(
                                $rb['engineCapacity']->greaterThanOrEqualTo(1),
                                $rb['engineCapacity']->lessThan(50)
                            )
                        ),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 8)
                    ]
                ),
                self::GROUP_SCOOTER_OTHER_ID
            )
        );

        // Type scooter of tot 50cc, minder dan 8.000km, jonger dan 11 jaar
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb->logicalOr(
                            $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                            $rb->logicalAnd(
                                $rb['engineCapacity']->greaterThanOrEqualTo(1),
                                $rb['engineCapacity']->lessThan(50)
                            )
                        ),
                        $rb['mileage']->lessThan(8000),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 11)
                    ]
                ),
                self::GROUP_SCOOTER_OTHER_ID
            )
        );

        // Overige scooters naar scooter schade en sloop
        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                    $rb->logicalAnd(
                        $rb['engineCapacity']->greaterThanOrEqualTo(1),
                        $rb['engineCapacity']->lessThan(50)
                    )
                ),
                self::GROUP_SCOOTER_DEMOLITION_ID
            )
        );

        // Voertuig type Trike of Quad -> Jong & Fris:
        $this->addRule(
            $rb->create(
                $rb['bodyType']->equalTo(BodyType::TRIKE),
                self::GROUP_YOUNG_ID
            )
        );

        // Voertuig type Klassier en merk BMW, Ducati, Benelli, Moto Guzzi, MZ of EML -> Overig
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    new LogicalOr(
                        [
                            $rb['modelName']->stringContainsInsensitive('BMW'),
                            $rb['modelName']->stringContainsInsensitive('DUCATI'),
                            $rb['modelName']->stringContainsInsensitive('BENELLI'),
                            $rb['modelName']->stringContainsInsensitive('MOTO GUZZI'),
                            $rb['modelName']->stringContainsInsensitive('MZ'),
                            $rb['modelName']->stringContainsInsensitive('EML'),
                        ]
                    ),
                    $rb['bodyType']->equalTo(BodyType::CLASSIC)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // Heeft het voertuig een zijspan -> Overige:
        $this->addRule(
            $rb->create(
                $rb['bodyType']->equalTo(BodyType::SIDE_CAR),
                self::GROUP_OTHER_ID
            )
        );

        // Is het voertuig van het merk Honda en bevat het model de tekst Goldwing of Gold wing
        // en is het voertuig jonger dan 15 jaar => Jong & Fris
        // Is het voertuig van het merk Harley Davison => Jong en Fris
        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    new LogicalAnd(
                        [
                            $rb['makeName']->equalTo('HONDA'),
                            $rb['modelName']->stringContainsInsensitive('GOLDWING'),
                            $rb['yearOfManufacture']->greaterThan($currentYear - 15)
                        ]
                    ),
                    $rb['makeName']->stringContainsInsensitive('HARLEY')
                ),
                self::GROUP_YOUNG_ID
            )
        );

        // Bevat het model de tekst Valkyrie, FJR, Wildstar of Midnight star en is het voertuig
        // ouder dan 9 jaar -> Overige
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    new LogicalOr(
                        [
                            $rb['modelName']->stringContainsInsensitive('VALKYRIE'),
                            $rb['modelName']->stringContainsInsensitive('FJR'),
                            $rb['modelName']->stringContainsInsensitive('WILDSTAR'),
                            $rb['modelName']->stringContainsInsensitive('MIDNIGHT STAR'),
                        ]
                    ),
                    $rb['yearOfManufacture']->lessThan($currentYear - 9)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // Bevat het model de tekst Goldwing of Gold wing en is het voertuig ouder dan 15 jaar -> Overig
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb->logicalOr(
                        $rb['modelName']->stringContainsInsensitive('GOLDWING'),
                        $rb['modelName']->stringContainsInsensitive('GOLD WING')
                    ),
                    $rb['yearOfManufacture']->lessThan($currentYear - 15)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // Is het voertuig van het merk BMW en heeft het minder dan 100.000km en is het een voertuig
        // tussen de 10 en 20 jaar oud -> Overige
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb['makeName']->equalTo('BMW'),
                        $rb['mileage']->lessThan(100000),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 20),
                        $rb['yearOfManufacture']->lessThan($currentYear - 10)
                    ]
                ),
                self::GROUP_OTHER_ID
            )
        );

        // Is het voertuig ouder dan 35 jaar -> Overig
        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->lessThan($currentYear - 35),
                self::GROUP_OTHER_ID
            )
        );

        // Is het voertuig jonger dan 9 jaar -> Jong & Fris
        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->greaterThan($currentYear - 9),
                self::GROUP_YOUNG_ID
            )
        );

        // Is het voertuig jonger dan 15 jaar en heeft het minder dan 70.000km -> Overige
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb['yearOfManufacture']->greaterThan($currentYear - 15),
                    $rb['mileage']->lessThan(70000)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // Is het voertuig jonger dan 15 jaar -> Budget
        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->greaterThan($currentYear - 15),
                self::GROUP_BUDGET_ID
            )
        );

        // Is het voertuig jonger dan 20 jaar en heeft het minder dan 40.000km -> Overige
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb['yearOfManufacture']->greaterThan($currentYear - 20),
                    $rb['mileage']->lessThan(40000)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // Is het voertuig jonger dan 20 jaar en heeft het minder dan 70.000km -> Budget
        // Is het voertuig jonger dan 25 jaar en heeft het minder dan 60.000km -> Budget
        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->greaterThan($currentYear - 20),
                        $rb['mileage']->lessThan(70000)
                    ),
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->greaterThan($currentYear - 25),
                        $rb['mileage']->lessThan(60000)
                    )
                ),
                self::GROUP_BUDGET_ID
            )
        );
        // Catch all
        $this->addRule(
            $rb->create(
                new CatchAll(),
                self::GROUP_DEMOLITION_ID
            )
        );
    }
}
