<?php

namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

class GroupOtherRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    public function __construct()
    {
        $currentYear = (int)date('Y');

        // Setup exclusions
        $excludedModelsRule = new Operator\LogicalNot([
            new Operator\LogicalOr([
                // Exclude Fresh&Young BMW's (< 40000 KM)
                new Operator\LogicalAnd([
                    ConditionUtils::equalTo('makeName', 'BMW'),
                    ConditionUtils::lessThan('mileage', 40000),
                ]),
                // Exclude BMW, and special models for mileage, years
                new Operator\LogicalAnd([
                    new Operator\LogicalOr([
                        ConditionUtils::equalTo('makeName', 'BMW'),
                        ConditionUtils::isOneOf('modelName', [
                            'PAN EUROPEAN', 'FJR', 'GOLDWING'
                        ]),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    ]),
                    new Operator\LogicalOr([
                        // Not within correct years:
                        ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 9),
                        ConditionUtils::lessThan('yearOfManufacture', $currentYear - 14),
                        // Not within correct mileage:
                        ConditionUtils::greaterThanOrEqualTo('mileage', 80000)
                    ])
                ]),
                // exclude all Harleys
                ConditionUtils::stringContains('makeName', 'HARLEY')
            ])
        ]);

        $this->groupRule = new Rule(
            new Operator\LogicalOr([
                // > 30000, 2017 - 2009
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 8),
                    ConditionUtils::greaterThan('mileage', 30000),
                    $excludedModelsRule
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 14),
                    ConditionUtils::lessThan('mileage', 50000),
                    $excludedModelsRule
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 19),
                    ConditionUtils::lessThan('mileage', 30000),
                    $excludedModelsRule
                )),
                // Exceptions: BMW, Pan European, FJR, Goldwing, Valkyrie
                // 2004 - 2009, only with mileage < 80K
                new Operator\LogicalAnd([
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    new Operator\LogicalOr([
                        ConditionUtils::equalTo('makeName', 'BMW'),
                        ConditionUtils::isOneOf('modelName', [
                            'PAN EUROPEAN', 'FJR', 'GOLDWING'
                        ]),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    ]),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 14),
                    ConditionUtils::lessThan('mileage', 80000),
                    $excludedModelsRule
                ]),
                // Classics
                ConditionUtils::equalTo('classic', true),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    new Operator\LogicalOr([
                        ConditionUtils::equalTo('classic', true),
                        ConditionUtils::lessThan('yearOfManufacture', 1980)
                    ]),
                    $excludedModelsRule
                )),
            ])
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_MOTORS_OTHER_ID;
    }
}
