<?php

namespace Dealerdirect\VehicleClassification\Tests;

use DealerDirect\Generic\Category\CallGroups;
use DealerDirect\Generic\Category\CallGroupsMoto;
use DealerDirect\Generic\Category\Site;
use Dealerdirect\VehicleClassification\VehicleClassificationIds;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\TestCase;

class VehicleClassificationIdsTest extends TestCase
{
    #[DataProvider('provide_valid_ids')]
    public function test_method_to_call_group_id_all(
        VehicleClassificationIds $vehicleClassificationId,
        int $reference,
        array $expected
    ): void {
        $this->assertEquals(
            $expected,
            $vehicleClassificationId->toCallGroupId($reference, true),
        );
    }

    #[DataProvider('provide_valid_ids')]
    public function test_method_to_call_group_id_single(
        VehicleClassificationIds $vehicleClassificationId,
        int $reference,
        array $expected
    ): void {
        $this->assertTrue(
            in_array(
                $vehicleClassificationId->toCallGroupId($reference),
                $expected,
                true,
            ),
            'Expected call group id not found in array of expected call group ids'
        );
    }

    #[DataProvider('provide_valid_ids')]
    public function test_value_returns_int(
        VehicleClassificationIds $vehicleClassificationId,
        int $reference,
        array $expected
    ): void {
        $this->assertIsInt(
            $vehicleClassificationId->value
        );
    }

    public function test_from_method(): void
    {
        $this->assertEquals(
            VehicleClassificationIds::CLASSIFICATION_NO_ID,
            VehicleClassificationIds::from(VehicleClassificationIds::CLASSIFICATION_NO_ID->value)
        );
    }

    public function test_try_from_method(): void
    {
        $this->expectException(\ValueError::class);

        VehicleClassificationIds::from(9999999);
    }

    public function test_to_call_group_id_should_throw_exception_when_invalid_reference_is_given(): void
    {
        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('No groups found for scooter NL classification id 51');

        VehicleClassificationIds::CLASSIFICATION_EXPORT_ID->toCallGroupId(Site::REF_SCOOTER_NL, false);
    }

    public static function provide_valid_ids(): array
    {
        return [
            [
                VehicleClassificationIds::CLASSIFICATION_NO_ID,
                Site::REF_CAR_NL,
                [
                    CallGroups::NL_CAR_GROUP_1,
                    CallGroups::NL_CAR_GROUP_2,
                    CallGroups::NL_CAR_GROUP_3,
                    CallGroups::NL_CAR_GROUP_4,
                ],
            ],
            [
                VehicleClassificationIds::CLASSIFICATION_DEMOLITION_ID,
                Site::REF_MOTO_NL,
                [
                    CallGroupsMoto::NL_MOTO_GROUP_1,
                    CallGroupsMoto::NL_MOTO_GROUP_2,
                    CallGroupsMoto::NL_MOTO_GROUP_3,
                    CallGroupsMoto::NL_MOTO_GROUP_4,
                ],
            ],
            [
                VehicleClassificationIds::CLASSIFICATION_EXPORT_ID,
                Site::REF_CAR_MARKTPLAATS_NL,
                [CallGroups::NL_CAR_MARKTPLAATS_GROUP_1],
            ],
        ];
    }
}
