<?php
namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Models\VehicleTypeMotorBodyType as BodyType;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Rules\VehicleRuleSet;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Rule;

class MotorsRuleSet extends VehicleRuleSet
{
    public function __construct()
    {
        $currentYear = (int)date('Y');
        // 1. Alles met een motorinhoud tot 50cc -> Brom & Snor:
        // 2. Voertuig type Bromfiets of Snorfiet -> Brom & Snor
        $this->add(
            new Rule(
                self::_OR_([
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_SCOOTER),
                    self::_AND_([
                        // Engine capacity less than 50 cc
                        ConditionUtils::greaterThanOrEqualTo('engineCapacity', 1),
                        ConditionUtils::lessThan('engineCapacity', 50),
                    ])
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_SCOOTER_ID
        );
        // 3. Voertuig type Trike of Quad -> Jong & Fris:
        $this->add(
            new Rule(
                self::_OR_([
                    ConditionUtils::equalTo('bodyType', BodyType::TRIKE),
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
        );
        // 4. Voertuig type Klassier en merk BMW, Ducati, Benelli, Moto Guzzi, MZ of EML -> Overig
        $this->add(
            new Rule(
                self::_AND_([
                    self::_OR_([
                        ConditionUtils::stringContains('modelName', 'BMW'),
                        ConditionUtils::stringContains('modelName', 'DUCATI'),
                        ConditionUtils::stringContains('modelName', 'BENELLI'),
                        ConditionUtils::stringContains('modelName', 'MOTO GUZZI'),
                        ConditionUtils::stringContains('modelName', 'MZ'),
                        ConditionUtils::stringContains('modelName', 'EML'),
                    ]),
                    ConditionUtils::equalTo('bodyType', BodyType::CLASSIC),
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );
        // 5. Heeft het voertuig een zijspan -> Overige:
        $this->add(
            new Rule(
                ConditionUtils::equalTo('bodyType', BodyType::SIDE_CAR)
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );
        // 6. Is het voertuig van het merk Honda en bevat het model de tekst Goldwing of Gold wing
        // en is het voertuig jonger dan 15 jaar => Jong & Fris
        // 7. Is het voertuig van het merk Harley Davison => Jong en Fris
        $this->add(
            new Rule(
                self::_OR_([
                    self::_AND_([
                        ConditionUtils::equalTo('makeName', 'HONDA'),
                        ConditionUtils::stringContains('modelName', 'GOLDWING'),
                        ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 15)
                    ]),
                    ConditionUtils::stringContains('makeName', 'HARLEY')
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
        );
        // 8. Bevat het model de tekst Valkyrie, FJR, Wildstar of Midnight star en is het voertuig
        // ouder dan 9 jaar -> Overige
        $this->add(
            new Rule(
                self::_AND_([
                    self::_OR_([
                        ConditionUtils::stringContains('modelName', 'VALKYRIE'),
                        ConditionUtils::stringContains('modelName', 'FJR'),
                        ConditionUtils::stringContains('modelName', 'WILDSTAR'),
                        ConditionUtils::stringContains('modelName', 'MIDNIGHT STAR'),
                    ]),
                    ConditionUtils::lessThan('yearOfManufacture', $currentYear - 9)
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );
        // 9. Bevat het model de tekst Goldwing of Gold wing en is het voertuig ouder dan 15 jaar -> Overig
        $this->add(
            new Rule(
                self::_AND_([
                    self::_OR_([
                        ConditionUtils::stringContains('modelName', 'GOLDWING'),
                        ConditionUtils::stringContains('modelName', 'GOLD WING'),
                    ]),
                    ConditionUtils::lessThan('yearOfManufacture', $currentYear - 15)
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );

        // 10. Is het voertuig van het merk BMW en heeft het minder dan 100.000km en is het een voertuig
        // tussen de 10 en 20 jaar oud -> Overige
        $this->add(
            new Rule(
                self::_AND_([
                    ConditionUtils::equalTo('makeName', 'BMW'),
                    ConditionUtils::lessThan('mileage', 100000),
                    ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 20),
                    ConditionUtils::lessThan('yearOfManufacture', $currentYear - 10)
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );
        // 11. Is het voertuig ouder dan 35 jaar -> Overig
        $this->add(
            new Rule(
                ConditionUtils::lessThan('yearOfManufacture', $currentYear - 35)
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );
        // 12. Is het voertuig jonger dan 9 jaar -> Jong & Fris
        $this->add(
            new Rule(
                ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 9)
            ),
            GroupRulesInterface::GROUP_MOTORS_YOUNG_ID
        );
        // 13. Is het voertuig jonger dan 15 jaar en heeft het minder dan 70.000km -> Overige
        $this->add(
            new Rule(
                self::_AND_([
                    ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::lessThan('mileage', 70000),
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );
        // 14. Is het voertuig jonger dan 15 jaar -> Budget
        $this->add(
            new Rule(
                ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 15)
            ),
            GroupRulesInterface::GROUP_MOTORS_BUDGET_ID
        );
        // 15. Is het voertuig jonger dan 20 jaar en heeft het minder dan 40.000km -> Overige
        $this->add(
            new Rule(
                self::_AND_([
                    ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 20),
                    ConditionUtils::lessThan('mileage', 40000),
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_OTHER_ID
        );
        // 16. Is het voertuig jonger dan 20 jaar en heeft het minder dan 70.000km -> Budget
        // 17. Is het voertuig jonger dan 25 jaar en heeft het minder dan 60.000km -> Budget
        $this->add(
            new Rule(
                self::_OR_([
                    self::_AND_([
                        ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 20),
                        ConditionUtils::lessThan('mileage', 70000),
                    ]),
                    self::_AND_([
                        ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 25),
                        ConditionUtils::lessThan('mileage', 60000),
                    ]),
                ])
            ),
            GroupRulesInterface::GROUP_MOTORS_BUDGET_ID
        );
    }

    /**
     * Evaluate context with specified rules in this ruleset
     * and return a resulting target group id
     *
     * When no match is found, GROUP_MOTORS_DEMOLITION_ID will be returned as default
     *
     * @param Context $context The vehicle context
     *
     * @return int The target group ID
     */
    public function evaluate(Context $context)
    {
        $resultGroupId = parent::evaluate($context);
        if ($resultGroupId === -1) {
            return GroupRulesInterface::GROUP_MOTORS_DEMOLITION_ID;
        }
        return $resultGroupId;
    }
}
