<?php
namespace DealerDirect\VehicleClassification\Rules;

use Ruler\Context;
use Ruler\Operator;

class VehicleRuleSet implements RuleSetInterface
{
    /**
     * @var array
     */
    protected $rules;

    /**
     * @var int Iterator index
     */
    private $index;

    public function __construct()
    {
        $this->rules = array();
        $this->index = -1;
    }

    public function add($rule, $targetGroupId)
    {
        $this->addRule(new GroupRule(
            $rule,
            $targetGroupId
        ));
    }

    /**
     * Add a Vehicle Group rule to the rule set
     * @param $rule GroupRule
     */
    public function addRule(GroupRule $rule)
    {
        $this->rules[] = $rule;
    }

    /**
     * Returns rule at given index in ruleset
     * @param $index
     * @return GroupRule
     */
    public function get($index)
    {
        if ($index >= sizeof($this->rules)) {
            throw new \OutOfBoundsException();
        }
        return $this->rules[$index];
    }

    /**
     * Returns next rule in the iteration, or null if there are no more
     *
     * @return null|GroupRule
     */
    public function next()
    {
        $this->index += 1;
        if ($this->index >= sizeof($this->rules)) {
            return null;
        }
        return $this->rules[$this->index];
    }

    /**
     * Reset iterator index
     */
    public function reset()
    {
        $this->index = -1;
    }

    /**
     * Evaluate context with specified rules in this ruleset
     * and return a resulting target group id
     * When no match is found, -1 will be returned
     *
     * @param Context $context The vehicle context
     * @return int The target group ID
     */
    public function evaluate(Context $context)
    {
        /**
         * @var $rule GroupRule
         */
        foreach ($this->rules as $rule) {
            if ($rule->evaluate($context)) {
                return $rule->getTargetGroupId();
            }
        }
        return -1;
    }

    /**
     * @param array $expression
     * @return Operator\LogicalAnd
     *
     * phpcs:disable PSR2.Methods.MethodDeclaration.Underscore,PSR1.Methods.CamelCapsMethodName.NotCamelCaps
     */
    public static function _AND_(array $expression)
    {
        return new Operator\LogicalAnd($expression);
    }

    public static function _OR_(array $expression)
    {
        return new Operator\LogicalOr($expression);
    }
}
