<?php

namespace DealerDirect\VehicleClassification\Models;

/**
 * Class Vehicle
 *
 * Main Vehicle class
 *
 * @package VehicleClassification\Model
 */
class Vehicle
{

    const TYPE_UNKNOWN = 0;

    const TYPE_CAR = 1;

    const TYPE_RECREATIONAL = 2;

    const TYPE_TRANSPORT = 3;

    const TYPE_MOTOR = 4;

    const TYPE_SCOOTER = 5;

    /**
     * Internal ID
     * @var int
     */
    private $id;

    /**
     * VIN Vehicle Identification Number
     * @var string
     */
    private $vin;

    /**
     * License plate number
     * var string
     */
    private $licensePlate;

    /**
     * Vehicle type
     * @var int
     */
    private $vehicleType;

    /**
     * Country the vehicle is based (sold from)
     * @var int
     */
    private $country;

    /**
     * Vehicle condition
     * @var int
     */
    private $condition = VehicleCondition::NOT_SPECIFIED;

    /**
     * Vehicle has damage?
     * @var bool
     */
    private $damageFree;

    /**
     * Is the vehicle drivable?
     * @var bool
     */
    private $drivable;

    /**
     * Vehicle mileage
     * @var int
     */
    private $mileage;

    /**
     * Make name
     * @var string
     */
    private $makeName;

    /**
     * Model name
     * @var string
     */
    private $modelName;

    /**
     * Year of vehicle manufacture
     * @var int
     */
    private $yearOfManufacture;

    /**
     * Type of fuel
     * @var string
     */
    private $fuelType;

    /**
     * @var int
     */
    private $bpm;

    /**
     * @var int
     */
    private $askingPrice;

    /**
     * @var int
     */
    private $catalogPrice;

    /**
     * @var int
     */
    private $valuationPrice;

    /**
     * True valuation of vehicle
     * @var int
     */
    private $actualValue;

    /**
     * Vehicle's door number
     * @var int
     */
    private $numberOfDoors;

    /**
     * Vehicle is a 'classic'?
     * @var bool
     */
    private $isClassic;

    /**
     * Vehicle options
     * @var array
     */
    private $options = array();

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @param int $id
     * @return Vehicle
     */
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * @return string
     */
    public function getVin()
    {
        return $this->vin;
    }

    /**
     * @param string $vin
     * @return Vehicle
     */
    public function setVin($vin)
    {
        $this->vin = $vin;

        return $this;
    }

    /**
     * @return string
     */
    public function getLicensePlate()
    {
        return $this->licensePlate;
    }

    /**
     * @param string $licensePlate
     * @return Vehicle
     */
    public function setLicensePlate($licensePlate)
    {
        $this->licensePlate = $licensePlate;

        return $this;
    }

    /**
     * @return int
     */
    public function getVehicleType()
    {
        return $this->vehicleType;
    }

    /**
     * @param int $vehicleType
     * @return Vehicle
     */
    public function setVehicleType($vehicleType)
    {
        $this->vehicleType = $vehicleType;

        return $this;
    }

    /**
     * @return int
     */
    public function getCountry()
    {
        return $this->country;
    }

    /**
     * @param int $country
     * @return Vehicle
     */
    public function setCountry($country)
    {
        $this->country = $country;

        return $this;
    }

    /**
     * @return int
     */
    public function getCondition()
    {
        return $this->condition;
    }

    /**
     * @param int $condition
     * @return Vehicle
     */
    public function setCondition($condition)
    {
        $this->condition = $condition;

        return $this;
    }

    /**
     * @return bool
     */
    public function isDamageFree()
    {
        return $this->damageFree;
    }

    /**
     * @param bool $damageFree
     * @return Vehicle
     */
    public function setDamageFree($damageFree)
    {
        $this->damageFree = $damageFree;

        return $this;
    }

    /**
     * @return bool
     */
    public function isDrivable()
    {
        return $this->drivable;
    }

    /**
     * @param bool $drivable
     * @return Vehicle
     */
    public function setDrivable($drivable)
    {
        $this->drivable = $drivable;

        return $this;
    }

    /**
     * @return int
     */
    public function getMileage()
    {
        return $this->mileage;
    }

    /**
     * @param int $mileage
     * @return Vehicle
     */
    public function setMileage($mileage)
    {
        $this->mileage = $mileage;

        return $this;
    }

    /**
     * @return string
     */
    public function getMakeName()
    {
        return $this->makeName;
    }

    /**
     * @param string $makeName
     * @return Vehicle
     */
    public function setMakeName($makeName)
    {
        $this->makeName = $makeName;

        return $this;
    }

    /**
     * @return string
     */
    public function getModelName()
    {
        return $this->modelName;
    }

    /**
     * @param string $modelName
     * @return Vehicle
     */
    public function setModelName($modelName)
    {
        $this->modelName = $modelName;

        return $this;
    }

    /**
     * @return int
     */
    public function getYearOfManufacture()
    {
        return $this->yearOfManufacture;
    }

    /**
     * @param int $yearOfManufacture
     * @return Vehicle
     */
    public function setYearOfManufacture($yearOfManufacture)
    {
        $this->yearOfManufacture = $yearOfManufacture;

        return $this;
    }

    /**
     * @return string
     */
    public function getFuelType()
    {
        return $this->fuelType;
    }

    /**
     * @param string $fuelType
     * @return $this
     */
    public function setFuelType($fuelType)
    {
        $this->fuelType = $fuelType;

        return $this;
    }

    /**
     * @return int
     */
    public function getBpm()
    {
        return $this->bpm;
    }

    /**
     * @param int $bpm
     * @return $this
     */
    public function setBpm($bpm)
    {
        $this->bpm = $bpm;

        return $this;
    }

    /**
     * @return int
     */
    public function getAskingPrice()
    {
        return $this->askingPrice;
    }

    /**
     * @param int $askingPrice
     * @return $this
     */
    public function setAskingPrice($askingPrice)
    {
        $this->askingPrice = $askingPrice;

        return $this;
    }

    /**
     * @return int
     */
    public function getCatalogPrice()
    {
        return $this->catalogPrice;
    }

    /**
     * @param int $catalogPrice
     * @return Vehicle
     */
    public function setCatalogPrice($catalogPrice)
    {
        $this->catalogPrice = $catalogPrice;

        return $this;
    }

    /**
     * @return int
     */
    public function getValuationPrice()
    {
        return $this->valuationPrice;
    }

    /**
     * @param int $valuationPrice
     * @return Vehicle
     */
    public function setValuationPrice($valuationPrice)
    {
        $this->valuationPrice = $valuationPrice;

        return $this;
    }

    /**
     * @return int
     */
    public function getActualValue()
    {
        return $this->actualValue;
    }

    /**
     * @param int $actualValue
     * @return Vehicle
     */
    public function setActualValue($actualValue)
    {
        $this->actualValue = $actualValue;

        return $this;
    }

    /**
     * @return int
     */
    public function getNumberOfDoors()
    {
        return $this->numberOfDoors;
    }

    /**
     * @param int $numberOfDoors
     * @return Vehicle
     */
    public function setNumberOfDoors($numberOfDoors)
    {
        $this->numberOfDoors = $numberOfDoors;

        return $this;
    }

    /**
     * Returns an array of the Vehicle's option values.
     *
     * For value definitions;
     * @see VehicleOptionInterface
     * @see CarOption
     * @see MotorOptionInterface
     * @see RecreationalOptionInterface
     *
     * @return array Integer array of vehicle's options
     */
    public function getOptions()
    {
        return $this->options;
    }

    /**
     * Whether the vehicle is considered a "classic"
     *
     * @return bool
     */
    public function isClassic()
    {
        return $this->isClassic;
    }

    /**
     * @param bool $isClassic
     * @return Vehicle
     */
    public function setClassic($isClassic)
    {
        $this->isClassic = $isClassic;

        return $this;
    }

    /**
     * @param array $options Integer Array of option values
     * @return Vehicle
     */
    public function setOptions(array $options)
    {
        // Only an array of int's is valid
        $this->options = $options;

        return $this;
    }

    /**
     * Check whether the vehicle has the given option [installed]
     *
     * @param int $option The option to check
     *
     * @return bool
     */
    public function hasOption($option)
    {
        return in_array($option, $this->options);
    }

    /**
     * Check whether the vehicle has ALL the given options [installed]
     *
     * @param array $options The option to check
     *
     * @return bool
     */
    public function hasAllOptions(array $options)
    {
        foreach ($options as $option) {
            if (!in_array($option, $this->options)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return <<<EOT
Vehicle ($this->id): 
    vehicleType: $this->vehicleType;
    licensePlate: $this->licensePlate;
    VIN: $this->vin;
    makeName: $this->makeName;
    modelName: $this->modelName;
EOT;
    }

}
