<?php

namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

/**
 * Motor Cycle "Budget Classification Group" Rules
 *
 * @package DealerDirect\VehicleClassification\Rules\Motors
 */
class GroupBudgetRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    public function __construct()
    {
        $currentYear = (int)date('Y');

        // Setup exclusions
        $excludedModelsRule = new Operator\LogicalNot([
            new Operator\LogicalOr([
                // Exclude Fresh&Young BMW's (< 40000 KM)
                new Operator\LogicalAnd([
                    ConditionUtils::equalTo('makeName', 'BMW'),
                    ConditionUtils::lessThan('mileage', 40000),
                ]),
                // Exclude BMW, and special models for mileage, years
                new Operator\LogicalAnd([
                    new Operator\LogicalOr([
                        ConditionUtils::equalTo('makeName', 'BMW'),
                        ConditionUtils::isOneOf('modelName', [
                            'PAN EUROPEAN', 'FJR', 'GOLDWING'
                        ]),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    ]),
                    new Operator\LogicalOr([
                        // Not within correct years:
                        ConditionUtils::greaterThan('yearOfManufacture', $currentYear - 15),
                        ConditionUtils::lessThan('yearOfManufacture', $currentYear - 20),
                        // Not within correct mileage:
                        ConditionUtils::greaterThanOrEqualTo('mileage', 100000)
                    ])
                ]),
                // Exclude all Harleys
                ConditionUtils::stringContains('makeName', 'HARLEY'),
                // Exclude classics
                ConditionUtils::equalTo('classic', true),
                ConditionUtils::lessThan('yearOfManufacture', 1980)
            ])
        ]);

        $this->groupRule = new Rule(
            new Operator\LogicalOr(array(
                // 2004 - 2009
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 14),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 50000),
                    $excludedModelsRule
                )),
                // 1999 - 2003
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 19),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 30000),
                    ConditionUtils::lessThan('mileage', 60000),
                    $excludedModelsRule
                )),
                // 1995 - 1999
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 20),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 23),
                    ConditionUtils::lessThan('mileage', 60000),
                    $excludedModelsRule
                )),
                // Exceptions: BMW, Pan European, FJR, Goldwing, Valkyrie
                // 1999 - 2003, only with mileage < 100K
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    new Operator\LogicalOr(array(
                        new Operator\LogicalAnd([
                            ConditionUtils::equalTo('makeName', 'BMW'),
                            ConditionUtils::greaterThanOrEqualTo('mileage', 40000),
                        ]),
                        ConditionUtils::isOneOf('modelName', ['PAN EUROPEAN', 'FJR']),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    )),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 19),
                    ConditionUtils::lessThan('mileage', 100000),
                )),
            ))
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_MOTORS_BUDGET_ID;
    }
}

