<?php

namespace DealerDirect\VehicleClassification\Rules\Motors;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

/**
 * Motor Cycle "Demolition Classification Group" Rules
 *
 * @package DealerDirect\VehicleClassification\Rules\Motors
 */
class GroupDemolitionRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    public function __construct()
    {
        $currentYear = (int)date('Y');

        $this->groupRule = new Rule(
            new Operator\LogicalOr(array(
                // Exceptions: BMW, Pan European, FJR, Goldwing, Valkyrie
                // only with mileage > 100K
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    new Operator\LogicalOr(array(
                        ConditionUtils::equalTo('makeName', 'BMW'),
                        ConditionUtils::isOneOf('modelName', [
                            'PAN EUROPEAN', 'FJR', 'GOLDWING'
                        ]),
                        ConditionUtils::stringContains('modelName', 'VALKYRIE')
                    )),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 100000),
                )),
                // Damaged motor cycles
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', false)
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 21),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 60000)
                )),
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_MOTOR),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 22),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', 1980)
                )),
            ))
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_MOTORS_DEMOLITION_ID;
    }
}

